--[[
	BaleCounter.lua
	
	Autor: 		Ifko[nator]
	Datum: 		21.04.2022
	Version: 	2.1
	
	Changelog:	v1.0 @26.12.2018 - initial implementation in FS 19
				--------------------------------------------------
				v2.0 @18.11.2021 - convert to FS 22
				--------------------------------------------------
				v2.1 @21.04.2022 - fix for patch 1.4 and higher
]]

BaleCounter = {};
BaleCounter.currentModName = "";

for _, mod in pairs(g_modManager.mods) do
	if mod.title == "Bale Counter" or mod.title == "Ballenzähler" or mod.title == "Compteur de balles pressées" then		
		if g_modIsLoaded[tostring(mod.modName)] then	
			BaleCounter.currentModName = mod.modName;
			
			break;
		end;
	end;
end;

function BaleCounter.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame;

	schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).baleCounter#countToday", "Bales pressed on field.", 0);
	schemaSavegame:register(XMLValueType.INT, "vehicles.vehicle(?).baleCounter#countTotal", "Bales pressed total.", 0);
end;

function BaleCounter.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Baler, specializations);
end;

function BaleCounter.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"saveToXMLFile",
		"onUpdate",
		"onRegisterActionEvents",
		"onWriteStream",
		"onReadStream"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, BaleCounter);
	end;
end;

function BaleCounter.registerFunctions(vehicleType)
	local newFunctions = {
		"setBaleCounterToZero"
	};
	
	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, BaleCounter[newFunction]);
	end;
end;

function BaleCounter:onLoad(savegame)
	local specBaleCounter = self.spec_baleCounter;
	
	if savegame ~= nil then
		specBaleCounter.countToday = savegame.xmlFile:getValue(savegame.key .. ".baleCounter#countToday", 0);
		specBaleCounter.countTotal = savegame.xmlFile:getValue(savegame.key .. ".baleCounter#countTotal", 0);
	else
		specBaleCounter.countToday = 0;
		specBaleCounter.countTotal = 0;
	end;
	
	if self.dropBale ~= nil then
		self.dropBale = Utils.prependedFunction(self.dropBale, BaleCounter.countDropBale);
	end;

	specBaleCounter.l10NTexts = {};

	local l10NTexts = {
		"PRESSED",
		"BALES_TODAY",
		"BALES_TOTAL",
		"input_RESET_BALE_COUNT_DAY_BUTTON"
	};

	for _, l10NText in pairs(l10NTexts) do
		specBaleCounter.l10NTexts[l10NText] = g_i18n:getText(l10NText, BaleCounter.modName);
	end;
end;

function BaleCounter:onRegisterActionEvents(isActiveForInput)
	if self.isClient then
        local specBaleCounter = self.spec_baleCounter;
        
		self:clearActionEventsTable(specBaleCounter.actionEvents);
        
		if self:getIsActiveForInput(true, false) then
			local _, actionEventId = self:addActionEvent(specBaleCounter.actionEvents, InputAction.RESET_BALE_COUNT_DAY_BUTTON, self, BaleCounter.actionEventResetBaleCountDay, false, true, false, true, nil);

			g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH);
			g_inputBinding:setActionEventTextVisibility(actionEventId, specBaleCounter.countToday > 0);
			g_inputBinding:setActionEventActive(actionEventId, specBaleCounter.countToday > 0);
		end;
	end;
end;

function BaleCounter.actionEventResetBaleCountDay(self, actionName, inputValue, callbackState, isAnalog)
	local specBaleCounter = self.spec_baleCounter;
	
	self:setBaleCounterToZero(0);
end;

function BaleCounter:setBaleCounterToZero(countToday, noEventSend)
	local specBaleCounter = self.spec_baleCounter;
	
	if countToday ~= specBaleCounter.countToday then
		if not noEventSend then
			if g_server ~= nil then
				g_server:broadcastEvent(BaleCounterEvent.new(self, countToday), nil, nil, self);
			else
				g_client:getServerConnection():sendEvent(BaleCounterEvent.new(self, countToday));
			end;
		end;
		
		specBaleCounter.countToday = countToday;
	end;
end;

function BaleCounter:onWriteStream(streamId, connection)
    if not connection:getIsServer() then 
		local specBaleCounter = self.spec_baleCounter;
		
		streamWriteInt32(streamId, specBaleCounter.countToday);
		streamWriteInt32(streamId, specBaleCounter.countTotal);
	end;
end;

function BaleCounter:onReadStream(streamId, connection)
    if connection:getIsServer() then
		local specBaleCounter = self.spec_baleCounter;
		
		specBaleCounter.countToday = streamReadInt32(streamId);
		specBaleCounter.countTotal = streamReadInt32(streamId);
	end;
end;

function BaleCounter:countDropBale(baleIndex)
	local specBaleCounter = self.spec_baleCounter;
	
	specBaleCounter.countToday = specBaleCounter.countToday + 1;
	specBaleCounter.countTotal = specBaleCounter.countTotal + 1;
end;

function BaleCounter:saveToXMLFile(xmlFile, key)
	local specBaleCounter = self.spec_baleCounter;
	
	xmlFile:setValue(key .. "#countToday", specBaleCounter.countToday);
	xmlFile:setValue(key .. "#countTotal", specBaleCounter.countTotal);
end;

function BaleCounter:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	if self:getIsActiveForInput(true, false) then
		local specBaleCounter = self.spec_baleCounter;
	
		g_currentMission:addExtraPrintText(specBaleCounter.l10NTexts.PRESSED .. " " .. specBaleCounter.l10NTexts.BALES_TODAY .. ": " .. specBaleCounter.countToday .. " | " .. specBaleCounter.l10NTexts.BALES_TOTAL .. ": " .. specBaleCounter.countTotal);

		local resetBaleCountDayButton = specBaleCounter.actionEvents[InputAction.RESET_BALE_COUNT_DAY_BUTTON];
	
		if resetBaleCountDayButton ~= nil then
			g_inputBinding:setActionEventActive(resetBaleCountDayButton.actionEventId, specBaleCounter.countToday > 0);
			g_inputBinding:setActionEventTextVisibility(resetBaleCountDayButton.actionEventId, specBaleCounter.countToday > 0);
			g_inputBinding:setActionEventText(resetBaleCountDayButton.actionEventId, specBaleCounter.l10NTexts.input_RESET_BALE_COUNT_DAY_BUTTON);
		end;
	end;
end;

BaleCounterEvent = {};
BaleCounterEvent_mt = Class(BaleCounterEvent, Event);

InitEventClass(BaleCounterEvent, "BaleCounterEvent");

function BaleCounterEvent.emptyNew()
	local self = Event.new(BaleCounterEvent_mt);
    
	return self;
end;

function BaleCounterEvent.new(baler, countToday)
	local self = BaleCounterEvent.emptyNew();
	
	self.baler = baler;
	self.countToday = countToday;
	
	return self;
end;

function BaleCounterEvent:readStream(streamId, connection)
	self.baler = NetworkUtil.readNodeObject(streamId);
	self.countToday = streamReadInt32(streamId);
    
	self:run(connection);
end;

function BaleCounterEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.baler);
	streamWriteInt32(streamId, self.countToday);
end;

function BaleCounterEvent:run(connection)
	if not connection:getIsServer() then
		g_server:broadcastEvent(BaleCounterEvent.new(self.baler, self.countToday), nil, connection, self.baler);
	end;
	
    if self.baler ~= nil then
        self.baler:setBaleCounterToZero(self.countToday, true);
	end;
end;