-- Recipe Changer for mixerWagon, allows to change between multiple recipes 
-- by modelleicher (Farming Agency)

mixerWagonRecipeChanger = {}

function mixerWagonRecipeChanger.registerOverwrittenFunctions(vehicleType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "addFillUnitFillLevel", mixerWagonRecipeChanger.addFillUnitFillLevel)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "getDischargeFillType", mixerWagonRecipeChanger.getDischargeFillType)
end

-- overwrite registerOverwrittenFunctions in default mixerWagon to disable overwriting of addFillUnitFillLevel and getDischargeFillType so we can do that instead 
function mixerWagonRecipeChanger.registerOverwrittenFunctionsMixerWagon(vehicleType)
	SpecializationUtil.registerOverwrittenFunction(vehicleType, "getFillUnitAllowsFillType", MixerWagon.getFillUnitAllowsFillType)
end
MixerWagon.registerOverwrittenFunctions = Utils.overwrittenFunction(MixerWagon.registerOverwrittenFunctions, mixerWagonRecipeChanger.registerOverwrittenFunctionsMixerWagon)


function mixerWagonRecipeChanger.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", mixerWagonRecipeChanger)
	
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", mixerWagonRecipeChanger)
end

function mixerWagonRecipeChanger.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "changeRecipe", mixerWagonRecipeChanger.changeRecipe)
end

-- onRegister actionEvent for MixerWagon
function mixerWagonRecipeChanger.onRegisterActionEvents(self, isActiveForInput, isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self.spec_mixerWagon

        self:clearActionEventsTable(spec.actionEvents)  

        if isActiveForInputIgnoreSelection and spec.currentRecipe ~= nil and spec.allowRecipeChanging then
            local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.IMPLEMENT_EXTRA3 , self, mixerWagonRecipeChanger.changeRecipeInput, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_HIGH)
            
            g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("action_mixerWagonToggleRecipe")..": "..g_i18n:getText("recipe_"..spec.outputFillType))
        end
    end
end

function mixerWagonRecipeChanger.getCurrentVehicleTypeHash(self, superFunc, vehicle)
	local vehicles = vehicle.rootVehicle.childVehicles
	local hash = ""
	for _, vehicle in pairs(vehicles) do
		hash = hash .. vehicle.typeName
	end
	return hash
end
InputHelpDisplay.getCurrentVehicleTypeHash = Utils.overwrittenFunction(InputHelpDisplay.getCurrentVehicleTypeHash, mixerWagonRecipeChanger.getCurrentVehicleTypeHash)


function mixerWagonRecipeChanger:changeRecipeInput()
	self:changeRecipe()
end


-- if vehicleType is changed by changeRecipe, joining as client on a dedi server ist no longer possible without restarting the server,
-- so we have to switch back the changed vehicleType "drivableMixerWagon<#>" to "drivableMixerWagon" to enable loading and joining
function mixerWagonRecipeChanger.overwrittenFunctionVehicleLoad(self, superfunc, vehicleData, asyncCallbackFunction, asyncCallbackObject, asyncCallbackArguments)
	if string.find(vehicleData.typeName, "drivableMixerWagon") and string.len(vehicleData.typeName)<=20 then
		vehicleData.typeName = "drivableMixerWagon"
	end
	if string.find(vehicleData.typeName, "mixerWagon") and string.len(vehicleData.typeName)<=12 then
		vehicleData.typeName = "mixerWagon"
	end
	superfunc(self, vehicleData, asyncCallbackFunction, asyncCallbackObject, asyncCallbackArguments)
end
Vehicle.load = Utils.overwrittenFunction(Vehicle.load, mixerWagonRecipeChanger.overwrittenFunctionVehicleLoad)


function mixerWagonRecipeChanger:changeRecipe(noEventSend, forceState, initialCall)

    local spec = self.spec_mixerWagon
	
    -- below a certain threshold switching is allowed, rest is discarded. This is because sometimes fillVolumes don't empty all the way and get stuck on a few liters left
    if self:getFillUnitFillLevel(spec.fillUnitIndex) <= 35 or (forceState ~= nil and forceState ~= spec.currentRecipe) then
	
        -- discard whatever is in currently
        self:addFillUnitFillLevel(self:getOwnerFarmId(), spec.fillUnitIndex, -math.huge, self:getFillUnitFillType(spec.fillUnitIndex), ToolType.UNDEFINED, nil)
		
        -- cycle through available recipes
		if g_currentMission ~= nil and g_currentMission.animalFoodSystem ~= nil and g_currentMission.animalFoodSystem.recipes ~= nil then
			local try = 1
			while true do
				-- select the next recipe
				spec.currentRecipe = spec.currentRecipe +1
				if spec.currentRecipe > #g_currentMission.animalFoodSystem.recipes then
					spec.currentRecipe = 1
				end
				-- see if the current recipe can be selected or not, break if we can select
				local recipe = g_currentMission.animalFoodSystem.recipes[spec.currentRecipe]
				if recipe.disableSelection == nil or recipe.disableSelection == false then
					break
				end

				-- in case we have only non-selectable break also.. 
				if try == #g_currentMission.animalFoodSystem.recipes then
					break
				end
				try = try +1
			end

			if forceState ~= nil then
				spec.currentRecipe = forceState
			end
			
		end
		
        -- "update" typeName to force vehicleHud update
		if not initialCall then -- only change typeName after vehicle is established not during initial loading
			self.typeName = tostring(spec.baseTypeName)..tostring(spec.currentRecipe)
		end
		
        -- reset recipe and reload
        spec.mixerWagonFillTypes = {}
        spec.fillTypeToMixerWagonFillType = {}
		
		if g_currentMission ~= nil and g_currentMission.animalFoodSystem ~= nil and g_currentMission.animalFoodSystem.recipes ~= nil then
			local recipe = g_currentMission.animalFoodSystem.recipes[spec.currentRecipe]
			
			spec.outputFillType = g_fillTypeManager:getFillTypeNameByIndex(recipe.fillType)

			for _, ingredient in ipairs(recipe.ingredients) do
				local entry = {}
				entry.fillLevel = 0
				entry.fillTypes = {}
				entry.fillTypesIndexed = {} 
				entry.name = ingredient.name
				entry.title = ingredient.title
				entry.minPercentage = ingredient.minPercentage
				entry.maxPercentage = ingredient.maxPercentage
				entry.ratio = ingredient.ratio

				for index, fillTypeIndex in ipairs(ingredient.fillTypes) do
					entry.fillTypes[fillTypeIndex] = true
					entry.fillTypesIndexed[index] = fillTypeIndex -- addition of different indexing method
					spec.fillTypeToMixerWagonFillType[fillTypeIndex] = entry
				end

				table.insert(spec.mixerWagonFillTypes, entry)
			end
		end
		
        if spec.actionEvents ~= nil and spec.actionEvents[InputAction.IMPLEMENT_EXTRA3] ~= nil then
            g_inputBinding:setActionEventText(spec.actionEvents[InputAction.IMPLEMENT_EXTRA3].actionEventId, g_i18n:getText("action_mixerWagonToggleRecipe")..": "..g_i18n:getText("recipe_"..spec.outputFillType))    
        end   
    end
	
	mixerWagonRecipeChangerEvent.sendEvent(self, spec.currentRecipe, noEventSend)	

end


function mixerWagonRecipeChanger:onLoad(savegame)

    local spec = self.spec_mixerWagon
    -- load recipe fillType as output fillType
    spec.outputFillType = Utils.getNoNil(string.upper(self.xmlFile:getValue("vehicle.mixerWagon#recipe", nil)), "FORAGE")
    spec.isForageOk = false 

    -- default Giants loading of recipe
    local recipeFillTypeName = self.xmlFile:getValue("vehicle.mixerWagon#recipe", "")
	
	-- only  allow for switching if recipe is default 
    if string.upper(recipeFillTypeName) == "FORAGE" then
		spec.allowRecipeChanging = true
	end
	
	-- we need to change vehicleTypeName for hudExtensions to update via changed hash.. 
	spec.baseTypeName = self.typeName     	
	
	-- convert recipe-fillType to recipe 
	local recipeFillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(string.upper(recipeFillTypeName))
	local recipe = g_currentMission.animalFoodSystem.recipeFillTypeIndexToRecipe[recipeFillTypeIndex]
	
	if recipe == nil then
		print("MaizePlus: Unable to find Recipe for FillType "..tostring(recipeFillTypeName).." , using default Recipe (1) instead")
	end

	-- intitialize variable for current recipe
	spec.currentRecipe = 1
	
	-- find recipe index for current recipe 
	for index, recipeFind in pairs(g_currentMission.animalFoodSystem.recipes) do
		if recipeFind == recipe then
			spec.currentRecipe = index 
		end
	end
	
	-- reset recipe and reload with custom additions 
	-- (This isn't neccessary as the default recipe should also be the default recipe, but we initialize fillTypesIndexed table and that way its also a backup in case somebody messes with the XML 
	spec.mixerWagonFillTypes = {}
	spec.fillTypeToMixerWagonFillType = {}

	-- reset local recipe according to default 1 or loaded above 
	recipe = g_currentMission.animalFoodSystem.recipes[spec.currentRecipe]
		
	if recipe == nil then
		Logging.xmlWarning(self.xmlFile, "MixerWagon recipe '%s' not defined!", recipeFillTypeName)
	end

	if recipe ~= nil then
		for _, ingredient in ipairs(recipe.ingredients) do
			local entry = {}
			entry.fillLevel = 0
			entry.fillTypes = {}
			entry.fillTypesIndexed = {} -- addition of this
			entry.name = ingredient.name
			entry.title = ingredient.title               
			entry.minPercentage = ingredient.minPercentage
			entry.maxPercentage = ingredient.maxPercentage
			entry.ratio = ingredient.ratio


			for index, fillTypeIndex in ipairs(ingredient.fillTypes) do
				entry.fillTypes[fillTypeIndex] = true
				entry.fillTypesIndexed[index] = fillTypeIndex -- addition of different indexing method
				spec.fillTypeToMixerWagonFillType[fillTypeIndex] = entry
			end

			table.insert(spec.mixerWagonFillTypes, entry)
		end
	end

end

-- prepend mixerWagon onPostLoad so recipe is set before fillLevels are loaded 
function mixerWagonRecipeChanger:onPostLoadPrepend(savegame)
    -- load current recipe first
    if savegame ~= nil then
        local spec = self.spec_mixerWagon
		
		if spec.currentRecipe ~= nil and spec.allowRecipeChanging then
			local currentRecipe = getXMLInt(savegame.xmlFile.handle, savegame.key..".FS22_MaizePlus.mixerWagonRecipeChanger#currentRecipe")
			
			if currentRecipe ~= nil and self.changeRecipe ~= nil then
				self:changeRecipe(true, currentRecipe, true)
			end
		end
    end
end
MixerWagon.onPostLoad = Utils.prependedFunction(MixerWagon.onPostLoad, mixerWagonRecipeChanger.onPostLoadPrepend)


function mixerWagonRecipeChanger:saveToXMLFile(xmlFile, key, usedModNames)
	local spec = self.spec_mixerWagon
	
    if spec.currentRecipe ~= nil and spec.allowRecipeChanging then
        setXMLInt(xmlFile.handle, key.."#currentRecipe", spec.currentRecipe)
    end
end


-- need to prepend onRead and onWrite to the base mixerWagon because it syncs mixerWagonFillTypes and those change depending on recipe so if server and client aren't on the same recipe it won't work 
-- so client and server need to synch recipe first 
function mixerWagonRecipeChanger:onReadStreamPrepend(streamId, connection)
	local spec = self.spec_mixerWagon

	if spec.currentRecipe ~= nil and spec.allowRecipeChanging then
		local recipe = streamReadUInt8(streamId)
		self:changeRecipe(true, recipe, true)
	end
end
MixerWagon.onReadStream = Utils.prependedFunction(MixerWagon.onReadStream, mixerWagonRecipeChanger.onReadStreamPrepend)

function mixerWagonRecipeChanger:onWriteStreamPrepend(streamId, connection)
	local spec = self.spec_mixerWagon

	if spec.currentRecipe ~= nil and spec.allowRecipeChanging then
		streamWriteUInt8(streamId, spec.currentRecipe)
	end
end
MixerWagon.onWriteStream = Utils.prependedFunction(MixerWagon.onWriteStream, mixerWagonRecipeChanger.onWriteStreamPrepend)


-- custom addFillUnitFillLevel largely based on the default one just with the addition of custom output filltypes
function mixerWagonRecipeChanger:addFillUnitFillLevel(superFunc, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
    
    local spec = self.spec_mixerWagon

    if fillUnitIndex ~= spec.fillUnitIndex then
        return superFunc(self, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
    end

    local oldFillLevel = self:getFillUnitFillLevel(fillUnitIndex)

    local mixerWagonFillType = spec.fillTypeToMixerWagonFillType[fillTypeIndex]

    -- allow to put forage back to mixer. Split it up into valid forage mixing ratios
    if fillTypeIndex == FillType[spec.outputFillType] and fillLevelDelta > 0 then  -- EDIT: change Forage to outputFilltype
        for _, entry in pairs(spec.mixerWagonFillTypes) do
            local delta = fillLevelDelta * entry.ratio
            self:addFillUnitFillLevel(farmId, fillUnitIndex, delta, next(entry.fillTypes), toolType, fillPositionData)
        end

        return fillLevelDelta
    end

    if mixerWagonFillType == nil then
        -- used for discharge
        if fillLevelDelta < 0 and oldFillLevel > 0 then
            -- remove values from all fill types such that the ratio doesn't change
            fillLevelDelta = math.max(fillLevelDelta, -oldFillLevel)

            local newFillLevel = 0
            for _, entry in pairs(spec.mixerWagonFillTypes) do
                local entryDelta = fillLevelDelta * (entry.fillLevel / oldFillLevel)
                entry.fillLevel = math.max(entry.fillLevel + entryDelta, 0)
                newFillLevel = newFillLevel + entry.fillLevel
            end

            if newFillLevel < 0.1 then
                for _, entry in pairs(spec.mixerWagonFillTypes) do
                    entry.fillLevel = 0
                end
                fillLevelDelta = -oldFillLevel
            end

            self:raiseDirtyFlags(spec.dirtyFlag)
            local ret = superFunc(self, farmId, fillUnitIndex, fillLevelDelta, fillTypeIndex, toolType, fillPositionData)
            return ret
        end

        return 0
    end

    local capacity = self:getFillUnitCapacity(fillUnitIndex)
    local free = capacity - oldFillLevel

    if fillLevelDelta > 0 then
        mixerWagonFillType.fillLevel = mixerWagonFillType.fillLevel + math.min(free, fillLevelDelta)
        if self:getIsSynchronized() then
            spec.activeTimer = spec.activeTimerMax
        end
    else
        mixerWagonFillType.fillLevel = math.max(0, mixerWagonFillType.fillLevel + fillLevelDelta)
    end

    local newFillLevel = 0
    for _, fillType in pairs(spec.mixerWagonFillTypes) do
        newFillLevel = newFillLevel + fillType.fillLevel
    end
    newFillLevel = MathUtil.clamp(newFillLevel, 0, self:getFillUnitCapacity(fillUnitIndex))

    local newFillType = FillType.UNKNOWN
    local isSingleFilled = false
    local isForageOk = false

    -- EDIT: add most fillType to get current fill when mixing not true
    local fillTypeMost = FillType.UNKNOWN
    local fillTypeMostLevel = 0
    for _, fillType in pairs(spec.mixerWagonFillTypes) do     
        if fillType.fillLevel > fillTypeMostLevel then
			-- get the first fillTypeIndex, fillType index is the numerical fillType representation (not sure if there is a better way?)
			for fillTypeIndex, _ in pairs(fillType.fillTypes) do
				fillTypeMost = fillTypeIndex
				break
			end
            fillTypeMostLevel = fillType.fillLevel
        end
    end   
    --

    for _, fillType in pairs(spec.mixerWagonFillTypes) do
        if newFillLevel == fillType.fillLevel then
            isSingleFilled = true
            newFillType = next(mixerWagonFillType.fillTypes)
            break
        end
    end

    if not isSingleFilled then
        isForageOk = true
        for _, fillType in pairs(spec.mixerWagonFillTypes) do      
            if fillType.fillLevel < fillType.minPercentage * newFillLevel - 0.01 or fillType.fillLevel > fillType.maxPercentage * newFillLevel + 0.01 then
                isForageOk = false
                break
            end
        end
    end

    if isForageOk then
        newFillType = FillType[spec.outputFillType]
    elseif not isSingleFilled then
        newFillType = fillTypeMost
    end
    spec.isForageOk = isForageOk

    self:raiseDirtyFlags(spec.dirtyFlag)

    self:setFillUnitFillType(fillUnitIndex, newFillType)


    return superFunc(self, farmId, fillUnitIndex, newFillLevel-oldFillLevel, newFillType, toolType, fillPositionData)
end

-- custom getDischargeFillType largely based on the default one just with the addition of custom output filltypes
function mixerWagonRecipeChanger:getDischargeFillType(superFunc, dischargeNode)
    local spec = self.spec_mixerWagon
    local fillUnitIndex = dischargeNode.fillUnitIndex

    if fillUnitIndex == spec.fillUnitIndex then
        local currentFillType = self:getFillUnitFillType(fillUnitIndex)
        local fillLevel = self:getFillUnitFillLevel(fillUnitIndex)

        --if currentFillType == FillType.FORAGE_MIXING and fillLevel > 0 then  
        if not spec.isForageOk then -- changed so it doesn't work through Forage_mixing anymore
            for _, entry in pairs(spec.mixerWagonFillTypes) do
                if entry.fillLevel > 0 then
                    currentFillType = next(entry.fillTypes)
                    break
                end
            end
        end

        return currentFillType, 1
    end

    return superFunc(self, dischargeNode)
end




mixerWagonRecipeChangerEvent = {}
local mixerWagonRecipeChangerEvent_mt = Class(mixerWagonRecipeChangerEvent, Event)

InitEventClass(mixerWagonRecipeChangerEvent, "mixerWagonRecipeChangerEvent")

function mixerWagonRecipeChangerEvent.emptyNew()
	local self = Event.new(mixerWagonRecipeChangerEvent_mt)
    self.className = "mixerWagonRecipeChangerEvent";
	return self
end

function mixerWagonRecipeChangerEvent.new(vehicle, recipe)
	local self = mixerWagonRecipeChangerEvent.emptyNew()
	self.vehicle = vehicle
	self.recipe = recipe
	
	return self
end

function mixerWagonRecipeChangerEvent:readStream(streamId, connection)
	self.vehicle = NetworkUtil.readNodeObject(streamId)
	self.recipe = streamReadInt8(streamId)

	self:run(connection)
end

function mixerWagonRecipeChangerEvent:writeStream(streamId, connection)
	NetworkUtil.writeNodeObject(streamId, self.vehicle)
	streamWriteInt8(streamId, self.recipe)
end

function mixerWagonRecipeChangerEvent:run(connection)

	if self.vehicle ~= nil and self.vehicle:getIsSynchronized() then
        self.vehicle:changeRecipe(true, self.recipe)
	end

	if not connection:getIsServer() then
		g_server:broadcastEvent(mixerWagonRecipeChangerEvent.new(self.vehicle, self.recipe), nil, connection, self.vehicle)
	end
end

function mixerWagonRecipeChangerEvent.sendEvent(vehicle, recipe, noEventSend)

	if (noEventSend == nil or noEventSend == false) then
		if g_server ~= nil then
			g_server:broadcastEvent(mixerWagonRecipeChangerEvent.new(vehicle, recipe), nil, nil, vehicle)
		else
			g_client:getServerConnection():sendEvent(mixerWagonRecipeChangerEvent.new(vehicle, recipe))
		end
	end
end


