--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 02.01.2022
Revision: FS22-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

InGameMenuObjectStorageFrame = {}

InGameMenuObjectStorageFrame.MOD_NAME = g_currentModName
InGameMenuObjectStorageFrame.MOD_DIR = g_currentModDirectory

InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER = 1
InGameMenuObjectStorageFrame.INFO_TEXT_STORAGE = 2
InGameMenuObjectStorageFrame.INFO_TEXT_GENERAL = 3

local InGameMenuObjectStorageFrame_mt = Class(InGameMenuObjectStorageFrame, TabbedMenuFrameElement)

local INVALID_NUMBER_TO_SPAWN = {"0 / 0"}
local EMPTY_TABLE = {}

InGameMenuObjectStorageFrame.CONTROLS = {
    "objectStoragesBox",
    "objectStorageList",
    "standardBox",
    "detailsLayout",
    "detailsLayoutName",
    "detailsLayoutIcon",
    "detailsLayoutSize",
    "numberToSpawnMulti",
    "storageStatusLayout",
    "storageStatusTitle",
    "statusObjectsText",
    "statusObjectsBar",
    "statusFermenting",
    "statusFermentingText",
    "statusFermentingBar",
    "statusNextObjectText",
    "statusNextObjectBar",
    "statusTotalText",
    "statusTotalBar",
    "fermentingHeader",
    "fermentingBox",
    "fermentingList",
    "fermentingListItem",
    "configurationBox",
    "configurationList",
    "infoBoxText",
    "noObjectStoragesBox"
}

function InGameMenuObjectStorageFrame.new(messageCenter, i18n)
    local self = InGameMenuObjectStorageFrame:superClass().new(nil, InGameMenuObjectStorageFrame_mt)

    self.i18n = i18n
    self.messageCenter = messageCenter

    self:registerControls(InGameMenuObjectStorageFrame.CONTROLS)

    self.hasCustomMenuButtons = true
    self.configurationActive = false

    self.hasInfoText = false
    self.updateTime = 0

    self.isMultiplayer = false
    self.spawnCoolDown = 0

    self.backButtonInfo = {
        inputAction = InputAction.MENU_BACK
    }

    self.menuButtonInfo = {
        self.backButtonInfo
    }

    self.infoTexts = {
        {text = "", updateTime = 0},
        {text = "", updateTime = 0},
        {text = "", updateTime = 0}
    }

    return self
end

function InGameMenuObjectStorageFrame:copyAttributes(src)
    InGameMenuObjectStorageFrame:superClass().copyAttributes(self, src)

    self.i18n = src.i18n
    self.messageCenter = src.messageCenter
end

function InGameMenuObjectStorageFrame:onGuiSetupFinished()
    InGameMenuObjectStorageFrame:superClass().onGuiSetupFinished(self)

    self.objectStorageList:setDataSource(self)
    self.fermentingList:setDataSource(self)
    self.configurationList:setDataSource(self)
end

function InGameMenuObjectStorageFrame:initialize()
    self.configurationStartButtonInfo = {
        profile = "buttonOk",
        inputAction = InputAction.MENU_ACCEPT,
        text = self.i18n:getText("shop_configuration"),
        callback = function ()
            self:onButtonConfigurationStart()
        end
    }

    self.configurationBackButtonInfo = {
        inputAction = InputAction.MENU_BACK,
        text = self.i18n:getText("button_cancel"),
        callback = function ()
            self:onButtonConfigurationBack()
        end
    }

    self.configurationConfirmButtonInfo = {
        profile = "buttonOk",
        inputAction = InputAction.MENU_ACCEPT,
        text = self.i18n:getText("button_confirm"),
        callback = function ()
            return self:onButtonConfigurationConfirm()
        end
    }

    self.spawnObjectsButtonInfo = {
        profile = "buttonOk",
        inputAction = InputAction.MENU_ACCEPT,
        text = self.i18n:getText("button_objectStorage_spawnObject", InGameMenuObjectStorageFrame.MOD_NAME),
        callback = function ()
            self:onButtonSpawnObjects()
        end
    }

    self.hotspotButtonInfo = {
        profile = "buttonHotspot",
        inputAction = InputAction.MENU_CANCEL,
        text = "Set Hotspot",
        callback = function ()
            self:onButtonHotspot()
        end
    }

    self.visitButtonInfo = {
        profile = "buttonVisitPlace",
        inputAction = InputAction.MENU_ACTIVATE,
        text = self.i18n:getText("action_visit"),
        callback = function ()
            self:onButtonVisit()
        end
    }

    self.inputTriggerButtonInfo = {
        profile = "buttonExtra1",
        inputAction = InputAction.MENU_EXTRA_1,
        text = "Input State",
        callback = function ()
            self:onButtonInputTrigger()
        end
    }

    self.cameraButtonInfo = {
        profile = "buttonExtra2",
        inputAction = InputAction.MENU_EXTRA_2,
        text = self.i18n:getText("ui_camera"),
        callback = function ()
            self:onButtonShowCamera()
        end
    }

    self.hotspotTagText = self.i18n:getText("action_tag")
    self.hotspotUntagText = self.i18n:getText("action_untag")

    self.storageEmptyText = self.i18n:getText("infohud_storageIsEmpty")
    self.storageFullText = self.i18n:getText("message_noSpaceForFillType")

    self.spawnAreaBlockedText = self.i18n:getText("message_objectStorage_spawnAreaBlocked", InGameMenuObjectStorageFrame.MOD_NAME)
    self.noSpawnSpaceText = self.i18n:getText("shop_messageNoSpace")

    self.enableInputText = self.i18n:getText("button_objectStorage_enableInput", InGameMenuObjectStorageFrame.MOD_NAME)
    self.disableInputText = self.i18n:getText("button_objectStorage_disableInput", InGameMenuObjectStorageFrame.MOD_NAME)

    self.fermentingBaleText = self.i18n:getText("infohud_bale") .. " %d"
end

function InGameMenuObjectStorageFrame:onFrameOpen()
    self.frameOpen = true

    InGameMenuObjectStorageFrame:superClass().onFrameOpen(self)

    self.isMultiplayer = g_currentMission.missionDynamicInfo.isMultiplayer
    self.objectStorageManager = g_objectStorageManager or g_currentMission.objectStorageManager

    self.messageCenter:subscribe("OBJECT_STORAGE_SPAWNER_STATUS", self.onSpawnerStatusChanged, self)
    self.messageCenter:subscribe("OBJECT_STORAGE_UPDATE_GUI", self.onUpdateRequest, self)

    self:updateVisibleFrames()
    self.objectStorageList:reloadData()

    FocusManager:setFocus(self.objectStorageList)
end

function InGameMenuObjectStorageFrame:onFrameClose()
    self.frameOpen = false

    self.messageCenter:unsubscribe("OBJECT_STORAGE_SPAWNER_STATUS", self)
    self.messageCenter:unsubscribe("OBJECT_STORAGE_UPDATE_GUI", self)

    self.selectedObjectStorage = nil
    self.selectedStorageArea = nil

    self.fermentingBales = nil

    self.configurationActive = false
    self.currentConfigurationIndex = nil

    self:updateInfoTexts(true)

    InGameMenuObjectStorageFrame:superClass().onFrameClose(self)
end

function InGameMenuObjectStorageFrame:update(dt)
    InGameMenuObjectStorageFrame:superClass().update(self, dt)

    if not self.configurationActive and g_currentMission.time > self.updateTime then
        self:reloadStorageListData(g_currentMission.time)
    end

    if self.hasInfoText then
        local raiseTextUpdate = false
        local missionTime = g_currentMission.time

        for i = 1, #self.infoTexts do
            local infoText = self.infoTexts[i]

            if infoText.text ~= "" and missionTime > infoText.updateTime then
                infoText.text = ""
                infoText.updateTime = 0

                raiseTextUpdate = true
            end
        end

        if raiseTextUpdate then
            self:updateInfoTexts(false)
        end
    end
end

function InGameMenuObjectStorageFrame:mouseEvent(posX, posY, isDown, isUp, button, eventUsed)
    if InGameMenuObjectStorageFrame:superClass().mouseEvent(self, posX, posY, isDown, isUp, button, eventUsed) then
        eventUsed = true
    end

    if isDown and not self.configurationActive then
        local element = self.numberToSpawnMulti

        -- If the mouse is overlapping the spawn spawner multiTextElement then allow the scroll wheel to adjust the values
        if GuiUtils.checkOverlayOverlap(posX, posY, element.absPosition[1], element.absPosition[2], element.absSize[1], element.absSize[2], nil) then
            local deltaIndex = 0

            if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_UP) then
                deltaIndex = 1
            elseif Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_DOWN) then
                deltaIndex = -1
            end

            if deltaIndex ~= 0 then
                element:setState(element.state + deltaIndex, true)

                eventUsed = true
            end
        end
    end

    return eventUsed
end

function InGameMenuObjectStorageFrame:setPlayerFarm(playerFarm)
    self.playerFarm = playerFarm

    if playerFarm ~= nil then
        self.configurationActive = false
        self.currentConfigurationIndex = nil

        self:updateVisibleFrames()

        if self.frameOpen then
            self:reloadStorageListData()
        end
    end
end

function InGameMenuObjectStorageFrame:setSelectedObjectStorage(objectStorage)
    local objectStorages = self:getObjectStorages()

    for i = 1, #objectStorages do
        if objectStorages[i] == objectStorage then
            self.objectStorageList:setSelectedItem(i, 1)

            break
        end
    end
end

function InGameMenuObjectStorageFrame:reloadStorageListData(missionTime, delayTime)
    missionTime = missionTime or g_currentMission.time

    if delayTime ~= nil then
        self.updateTime = missionTime + delayTime

        return
    end

    self.updateTime = missionTime + 5000

    -- Correct focus if required ??
    local correctFocus = self.numberToSpawnMulti.focusActive

    self.numberToSpawnMulti.focusActive = false
    self.objectStorageList:reloadData()

    if correctFocus then
        self.numberToSpawnMulti:setSoundSuppressed(true)
        FocusManager:setFocus(self.numberToSpawnMulti)
        self.numberToSpawnMulti:setSoundSuppressed(false)
    end
end

function InGameMenuObjectStorageFrame:updateVisibleFrames()
    self.lastNumSections = #self:getObjectStorages()

    local hasObjectStorages = self.lastNumSections > 0
    local configActive = self.configurationActive

    self.configurationBox:setVisible(configActive)
    self.standardBox:setVisible(not configActive)

    self.objectStoragesBox:setVisible(hasObjectStorages)
    self.noObjectStoragesBox:setVisible(not hasObjectStorages)

    if not hasObjectStorages then
        self:updateMenuButtons()
    end
end

function InGameMenuObjectStorageFrame:updateMenuButtons()
    self.menuButtonInfo = {
        self.backButtonInfo
    }

    local objectStorage, storageArea = self:getCurrentSelection()

    if storageArea ~= nil then
        if storageArea.numObjects > 0 then
            local spawnSpace, spawnAreaBlocked = objectStorage:getMaxSpawnSpace(storageArea, 1)

            self.spawnObjectsButtonInfo.disabled = spawnAreaBlocked or spawnSpace == 0
            self.menuButtonInfo[2] = self.spawnObjectsButtonInfo
        elseif self.configurationActive then
            self.menuButtonInfo[1] = self.configurationBackButtonInfo
            self.menuButtonInfo[2] = self.configurationConfirmButtonInfo
        elseif objectStorage:getAcceptedType(2, storageArea.sortedAcceptedTypes) ~= nil then
            self.menuButtonInfo[2] = self.configurationStartButtonInfo
        end

        if not self.configurationActive then
            local areaHotspot = objectStorage:getHotspot(storageArea)

            if areaHotspot ~= nil then
                self.hotspotButtonInfo.text = self.hotspotTagText

                if g_currentMission.currentMapTargetHotspot == areaHotspot then
                    self.hotspotButtonInfo.text = self.hotspotUntagText
                end

                table.insert(self.menuButtonInfo, self.hotspotButtonInfo)

                if areaHotspot.getBeVisited and areaHotspot:getBeVisited() then
                    table.insert(self.menuButtonInfo, self.visitButtonInfo)
                end
            end

            if objectStorage.canDisableInputTrigger then
                self.inputTriggerButtonInfo.text = self.enableInputText

                if objectStorage.inputTriggerState then
                    self.inputTriggerButtonInfo.text = self.disableInputText
                end

                table.insert(self.menuButtonInfo, self.inputTriggerButtonInfo)
            end

            if storageArea.cameraNode ~= nil then
                table.insert(self.menuButtonInfo, self.cameraButtonInfo)
            end
        end
    end

    self:setMenuButtonInfoDirty()
end

function InGameMenuObjectStorageFrame:populateCellForItemInSection(list, section, index, cell)
    if list == self.objectStorageList then
        local objectStorage = self:getObjectStorage(section)

        local imageFilename = "dataS/menu/hud/fillTypes/hud_fill_unknown.dds"
        local nameText = "Unknown"
        local numObjects = 0

        if objectStorage.indexedStorageAreas ~= nil then
            local storageArea = objectStorage.indexedStorageAreas[index]

            if storageArea ~= nil and storageArea.fillTypeIndex then
                local objectType = storageArea.objectType

                imageFilename = self.objectStorageManager:getHudOverlayFilename(storageArea.fillTypeIndex, objectStorage.baleStorage, objectType.isRoundbale)
                nameText = g_fillTypeManager:getFillTypeTitleByIndex(storageArea.fillTypeIndex)

                if objectStorage.baleStorage then
                    local size = (objectType.isRoundbale and storageArea.objectType.diameter or storageArea.objectType.length) or 0

                    nameText = string.format("%s (%d cm)", nameText, size * 100)
                end

                numObjects = #storageArea.objects
            end
        end

        cell:getAttribute("icon"):setImageFilename(imageFilename)
        cell:getAttribute("name"):setText(nameText)
        cell:getAttribute("numObjects"):setText(tostring(numObjects))
    elseif list == self.fermentingList then
        local fermentingBale = self.fermentingBales and self.fermentingBales[index]

        if fermentingBale ~= nil then
            cell:getAttribute("index"):setText(string.format(self.fermentingBaleText, fermentingBale.index))
            -- cell:getAttribute("percent"):setValue(fermentingBale.fermentingPercentage)
            cell:getAttribute("percent"):setText(string.format("%d%%", fermentingBale.fermentingPercentage * 100))
        else
            cell:getAttribute("index"):setText("Unknown")
            cell:getAttribute("percent"):setText("0%")
        end
    elseif list == self.configurationList then
        local sizeText = ""
        local nameText = "Unknown"
        local imageFilename = "dataS/menu/hud/fillTypes/hud_fill_unknown.dds"

        local acceptedType = nil
        local objectStorage, storageArea = self:getCurrentSelection()

        if objectStorage.getAcceptedType ~= nil then
            acceptedType = objectStorage:getAcceptedType(index, storageArea ~= nil and storageArea.sortedAcceptedTypes or nil)
        end

        if acceptedType ~= nil and acceptedType.fillTypeIndex ~= nil and acceptedType.objectType ~= nil then
            local objectType = acceptedType.objectType

            nameText = g_fillTypeManager:getFillTypeTitleByIndex(acceptedType.fillTypeIndex)
            imageFilename = self.objectStorageManager:getHudOverlayFilename(acceptedType.fillTypeIndex, objectStorage.baleStorage, objectType.isRoundbale)

            if objectStorage.baleStorage then
                if objectType.isRoundbale then
                    sizeText = string.format("%.2f x %.2f", (objectType.width or 0), (objectType.diameter or 0))
                    nameText = string.format("%s (%d cm)", nameText, (objectType.diameter or 0) * 100)
                else
                    sizeText = string.format("%.2f x %.2f x %.2f", (objectType.width or 0), (objectType.height or 0), (objectType.length or 0))
                    nameText = string.format("%s (%d cm)", nameText, (objectType.length or 0) * 100)
                end
            end
        end

        cell:getAttribute("size"):setText(sizeText)
        cell:getAttribute("name"):setText(nameText)
        cell:getAttribute("icon"):setImageFilename(imageFilename)
    end
end

function InGameMenuObjectStorageFrame:onListSelectionChanged(list, section, index)
    if list == self.objectStorageList then
        local objectStorage = self:getObjectStorage(section)

        local storageArea = nil
        local objectStorageName = ""
        local numberToSpawnTexts = INVALID_NUMBER_TO_SPAWN

        local validArea = false
        local resetSpawnSelector = false
        local storageAreaFillType = FillType.UNKNOWN

        if objectStorage ~= nil and objectStorage.indexedStorageAreas ~= nil then
            storageArea = objectStorage.indexedStorageAreas[index]

            if storageArea ~= nil and storageArea.fillTypeIndex then
                storageAreaFillType = storageArea.fillTypeIndex
            end
        end

        if self.selectedObjectStorage ~= objectStorage or self.selectedStorageArea ~= storageArea then
            self:updateInfoTexts(true) -- clear all texts
            resetSpawnSelector = true
        end

        self.selectedObjectStorage = objectStorage
        self.selectedStorageArea = storageArea

        self.fermentingBales = nil

        if storageAreaFillType ~= FillType.UNKNOWN and storageArea.objectType ~= nil then
            local numObjects = #storageArea.objects
            local maxObjects = storageArea.maxObjects

            local objectType = storageArea.objectType
            local baleStorage = objectStorage.baleStorage
            local isRoundbale = objectType.isRoundbale

            local sizeText = ""
            local unitShort = objectType.unitShort or "l"

            if unitShort == "" then
                unitShort = "l"
            end

            local nextObjectText = "0 " .. unitShort

            local totalFillLevel = 0
            local totalCapacity = 0

            local objectsBarValue = 0
            local nextObjectBarValue = 0
            local totalBarValue = 0

            local showFermentingStatus = false
            local showFermentingData = false

            objectStorageName = objectStorage:getName()

            self.detailsLayoutName:setText(g_fillTypeManager:getFillTypeTitleByIndex(storageAreaFillType))
            self.detailsLayoutIcon:setImageFilename(self.objectStorageManager:getHudOverlayFilename(storageAreaFillType, objectStorage.baleStorage, objectType.isRoundbale))

            if objectStorage.baleStorage then
                if objectType.isRoundbale then
                    sizeText = string.format("%.2f x %.2f", objectType.width or 0, objectType.diameter or 0)
                else
                    sizeText = string.format("%.2f x %.2f x %.2f", objectType.width or 0, objectType.height or 0, objectType.length or 0)
                end

                if storageArea.supportsFermenting and objectStorage.getFermentingBales ~= nil then
                    self.fermentingBales = objectStorage:getFermentingBales(storageArea)
                end

                if self.fermentingBales ~= nil then
                    local numFermenting = #self.fermentingBales
                    local fermentingBarValue = 0

                    showFermentingStatus = true

                    if numFermenting > 0 and numObjects > 0 then
                        fermentingBarValue = numFermenting / numObjects
                        showFermentingData = true
                    end

                    self.statusFermentingText:setText(string.format("%d / %d", numFermenting, numObjects))
                    self:updateStatusBar(self.statusFermentingBar, fermentingBarValue, false)

                    self.fermentingList:reloadData()
                end
            end

            self.detailsLayoutSize:setText(sizeText)

            if maxObjects > 0 then
                objectsBarValue = numObjects / maxObjects
            end

            if numObjects > 0 then
                local maxSpawnSpace, spawnAreaBlocked, objectName = objectStorage:getMaxSpawnSpace(storageArea, numObjects)
                local canSpawn = false

                if numObjects >= maxObjects then
                    self:addInfoText(self.storageFullText, InGameMenuObjectStorageFrame.INFO_TEXT_STORAGE, false)
                else
                    self:addInfoText("", InGameMenuObjectStorageFrame.INFO_TEXT_STORAGE, false)
                end

                if spawnAreaBlocked then
                    if objectName == nil or objectName == "" then
                        objectName = "Unknown"
                    end

                    self:addInfoText(string.format(self.spawnAreaBlockedText, objectName), InGameMenuObjectStorageFrame.INFO_TEXT_GENERAL, true)
                elseif maxSpawnSpace > 0 then
                    numberToSpawnTexts = {}
                    canSpawn = true

                    self:addInfoText("", InGameMenuObjectStorageFrame.INFO_TEXT_GENERAL, true)
                elseif self.infoTexts[InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER] ~= self.noSpawnSpaceText then
                    self:addInfoText(self.noSpawnSpaceText, InGameMenuObjectStorageFrame.INFO_TEXT_GENERAL, true)
                end

                local firstObject = storageArea.objects[numObjects]
                local objectFillLevel = firstObject.fillLevel
                local objectCapacity = firstObject.capacity

                if objectCapacity == nil or objectCapacity == 0 then
                    objectCapacity = objectFillLevel
                end

                nextObjectText = self.i18n:formatVolume(objectFillLevel, 0, unitShort)
                nextObjectBarValue = objectFillLevel / objectCapacity

                for i = 1, numObjects do
                    local object = storageArea.objects[i]
                    local capacity = object.capacity

                    if capacity == nil or capacity == 0 then
                        capacity = object.fillLevel
                    end

                    totalFillLevel = totalFillLevel + object.fillLevel
                    totalCapacity = totalCapacity + capacity

                    if canSpawn and i <= maxSpawnSpace then
                        numberToSpawnTexts[i] = string.format("%d  /  %d", i, maxSpawnSpace)
                    end
                end

                if totalCapacity > 0 then
                    totalBarValue = totalFillLevel / totalCapacity
                end
            else
                self:addInfoText(self.storageEmptyText, InGameMenuObjectStorageFrame.INFO_TEXT_STORAGE, true)
            end

            self.statusObjectsText:setText(string.format("%d / %d", numObjects, maxObjects))
            self:updateStatusBar(self.statusObjectsBar, objectsBarValue, true, true)

            self.statusNextObjectText:setText(nextObjectText)
            self:updateStatusBar(self.statusNextObjectBar, nextObjectBarValue, true, false)

            self.statusTotalText:setText(self.i18n:formatVolume(totalFillLevel, 0, unitShort))
            self:updateStatusBar(self.statusTotalBar, totalBarValue, true, false)

            self.statusFermenting:setVisible(showFermentingStatus)
            self.fermentingHeader:setVisible(showFermentingData)
            self.fermentingBox:setVisible(showFermentingData)

            validArea = true
        end

        self.storageStatusTitle:setText(objectStorageName)

        self.detailsLayout:setVisible(validArea)
        self.storageStatusLayout:setVisible(validArea)

        self.numberToSpawnMulti:setTexts(numberToSpawnTexts)
        self.numberToSpawnMulti:setDisabled(not validArea)

        if validArea then
            if resetSpawnSelector then
                self.numberToSpawnMulti:setState(1)
            end

            self.storageStatusLayout:invalidateLayout()
        else
            self.numberToSpawnMulti:setState(1)
        end

        if self.configurationActive then
            self.updateTime = g_currentMission.time + 5000

            self.configurationActive = false
            self.currentConfigurationIndex = nil
            self.configurationConfirmButtonInfo.disabled = false

            self:updateVisibleFrames()
        end

        self:updateMenuButtons()
    elseif list == self.configurationList then
        local disabled = false

        if self.currentConfigurationIndex ~= nil then
            disabled = self.currentConfigurationIndex == index
        end

        self.configurationConfirmButtonInfo.disabled = disabled
        self:updateMenuButtons()
    end
end

function InGameMenuObjectStorageFrame:updateStatusBar(element, barValue, allowBarProfileChange, lowIsGood)
    local profile = "ingameMenuObjectStorageStatusBar"

    if allowBarProfileChange then
        if (lowIsGood and barValue > 0.7) or (not lowIsGood and barValue < 0.3) then
            profile = "ingameMenuObjectStorageStatusBarLow"
        elseif barValue >= 0.3 and barValue <= 0.7 then
            profile = "ingameMenuObjectStorageStatusBarMedium"
        end
    end

    local barFullWidth = element.parent.size[1] - element.margin[1] * 2
    local minimumBarSize = 0

    if element.startSize ~= nil then
        minimumBarSize = element.startSize[1] + element.endSize[1]
    end

    element:applyProfile(profile)
    element:setSize(math.max(minimumBarSize, barFullWidth * math.min(1, barValue)), nil)
end

function InGameMenuObjectStorageFrame:getNumberOfSections(list, section)
    if list == self.objectStorageList then
        local numSections = #self:getObjectStorages()

        if numSections ~= self.lastNumSections then
            self.lastNumSections = numSections

            self.selectedObjectStorage = nil
            self.selectedStorageArea = nil

            self:updateVisibleFrames()
        end

        return numSections
    end

    return 1
end

function InGameMenuObjectStorageFrame:getNumberOfItemsInSection(list, section)
    if list == self.objectStorageList then
        local objectStorage = self:getObjectStorage(section)

        if objectStorage.indexedStorageAreas ~= nil then
            return #objectStorage.indexedStorageAreas
        end
    elseif list == self.fermentingList then
        if self.fermentingBales ~= nil then
            return #self.fermentingBales
        end
    elseif list == self.configurationList then
        local objectStorage, storageArea = self:getCurrentSelection()

        if storageArea ~= nil and storageArea.sortedAcceptedTypes ~= nil then
            return #storageArea.sortedAcceptedTypes
        end

        if objectStorage.sortedAcceptedTypes ~= nil then
            return #objectStorage.sortedAcceptedTypes
        end
    end

    return 0
end

function InGameMenuObjectStorageFrame:populateSectionHeader(list, section, element)
    if list == self.objectStorageList then
        local objectStorage = self:getObjectStorage(section)

        if objectStorage ~= nil then
            local titleElement = element:getAttribute("title")
            local inputStateElement = element:getAttribute("inputState")

            titleElement:setText(objectStorage:getName())

            if objectStorage.inputTriggerState then
                inputStateElement:applyProfile("ingameMenuObjectStorageInputEnabled")
            else
                inputStateElement:applyProfile("ingameMenuObjectStorageInputDisabled")
            end
        end
    end
end

function InGameMenuObjectStorageFrame:getObjectStorages()
    if self.playerFarm == nil or self.objectStorageManager == nil then
        return EMPTY_TABLE
    end

    return self.objectStorageManager:getFarmStorages(self.playerFarm.farmId)
end

function InGameMenuObjectStorageFrame:getObjectStorage(section)
    return self:getObjectStorages()[section] or EMPTY_TABLE
end

function InGameMenuObjectStorageFrame:getCurrentSelection()
    local objectStorage, storageArea = self:getObjectStorage(self.objectStorageList:getSelectedSection()), nil

    if objectStorage.indexedStorageAreas ~= nil then
        storageArea = objectStorage.indexedStorageAreas[self.objectStorageList:getSelectedIndexInSection()]
    end

    return objectStorage, storageArea
end

function InGameMenuObjectStorageFrame:onUpdateRequest(objectStorage, forceUpdate)
    if self.frameOpen then
        if forceUpdate == true or self:getObjectStorage(self.objectStorageList:getSelectedSection()) == objectStorage then
            self:reloadStorageListData()
        end
    end
end

function InGameMenuObjectStorageFrame:onSpawnerStatusChanged(objectStorage, statusCode)
    if self.frameOpen then
        local spawnCoolDown, delayTime = 0, nil

        if self:getObjectStorage(self.objectStorageList:getSelectedSection()) == objectStorage then
            if statusCode == ObjectStorageSpawner.RESULT_SUCCESS then
                local spawnSuccessText = self.i18n:getText("message_objectStorage_spawnObjectSuccess", InGameMenuObjectStorageFrame.MOD_NAME)

                if objectStorage.baleStorage then
                    self:addInfoText(string.format(spawnSuccessText, self.i18n:getText("category_bales")), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                else
                    self:addInfoText(string.format(spawnSuccessText, self.i18n:getText("category_pallets")), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                end
            elseif statusCode == ObjectStorageSpawner.RESULT_PART_SUCCESS then
                local spawnFailText = self.i18n:getText("message_objectStorage_spawnObjectFail", InGameMenuObjectStorageFrame.MOD_NAME)

                if objectStorage.baleStorage then
                    self:addInfoText(string.format(spawnFailText, self.i18n:getText("category_bales")), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                else
                    self:addInfoText(string.format(spawnFailText, self.i18n:getText("category_pallets")), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                end
            elseif statusCode == ObjectStorageSpawner.RESULT_NO_SPACE then
                if self.infoTexts[InGameMenuObjectStorageFrame.INFO_TEXT_GENERAL] ~= self.noSpawnSpaceText then
                    self:addInfoText(self.noSpawnSpaceText, InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                end
            elseif statusCode == ObjectStorageSpawner.OBJECT_LIMIT_REACHED then
                if objectStorage.baleStorage then
                    self:addInfoText(self.i18n:getText("warning_tooManyBales"), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                else
                    self:addInfoText(self.i18n:getText("warning_tooManyPallets"), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)
                end
            end
        end

        if self.isMultiplayer then
            local remainingCoolDown = math.max(self.spawnCoolDown - g_currentMission.time, 0)

            if remainingCoolDown > 0 then
                spawnCoolDown = self.spawnCoolDown
                delayTime = remainingCoolDown
            end
        end

        self:reloadStorageListData(g_currentMission.time, delayTime)
        self.spawnCoolDown = spawnCoolDown
    end
end

function InGameMenuObjectStorageFrame:onButtonSpawnObjects(element)
    -- Cool down is mainly for MP to give the server a chance and also to discourage pressing spawn button many times rather then selecting required amount.
    if self.spawnCoolDown < g_currentMission.time then
        local objectStorage, storageArea = self:getCurrentSelection()

        if storageArea ~= nil and storageArea.numObjects > 0 then
            objectStorage:spawnObjects(storageArea.index, self.numberToSpawnMulti:getState())
            self.spawnCoolDown = g_currentMission.time + 500
        end
    else
        self:playSample(GuiSoundPlayer.SOUND_SAMPLES.ERROR)
        self:addInfoText(self.i18n:getText("message_objectStorage_spawnCoolDown", InGameMenuObjectStorageFrame.MOD_NAME), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true, 2000)
    end
end

function InGameMenuObjectStorageFrame:onButtonHotspot()
    local objectStorage, storageArea = self:getCurrentSelection()

    if objectStorage ~= nil then
        local areaHotspot = objectStorage:getHotspot(storageArea)

        if areaHotspot ~= nil then
            if g_currentMission.currentMapTargetHotspot == areaHotspot then
                g_currentMission:setMapTargetHotspot(nil)
            else
                g_currentMission:setMapTargetHotspot(areaHotspot)
            end

            self:updateMenuButtons()
        else
            g_currentMission:setMapTargetHotspot(nil)
        end
    end
end

function InGameMenuObjectStorageFrame:onButtonVisit()
    local objectStorage, storageArea = self:getCurrentSelection()

    if objectStorage ~= nil then
        local areaHotspot = objectStorage:getHotspot(storageArea)

        if areaHotspot ~= nil then
            local x, y, z = areaHotspot:getTeleportWorldPosition()

            if x ~= nil and y ~= nil and z ~= nil then
                if g_currentMission.controlledVehicle ~= nil then
                    g_currentMission:onLeaveVehicle(x, y, z, true, false)
                else
                    g_currentMission.player:moveToAbsolute(x, y, z, false, false)
                end

                g_gui:changeScreen(nil)
            end
        end
    end
end

function InGameMenuObjectStorageFrame:onButtonInputTrigger()
    local objectStorage, _ = self:getCurrentSelection()

    if objectStorage ~= nil and objectStorage.setInputTriggerState ~= nil then
        objectStorage:setInputTriggerState(not objectStorage.inputTriggerState)
        self:reloadStorageListData()
    end
end

function InGameMenuObjectStorageFrame:onButtonShowCamera()
    local objectStorage, storageArea = self:getCurrentSelection()

    if storageArea ~= nil and storageArea.cameraNode ~= nil then
        -- Apply camera and header before opening
        local dialog = g_gui.guis.CameraDialog

        if dialog ~= nil then
            dialog.target:createCameraOverlay(storageArea.cameraNode)
            dialog.target:setHeaderText(objectStorage:getName())

            g_gui:showDialog("CameraDialog")
        end
    end
end

function InGameMenuObjectStorageFrame:onButtonConfigurationStart()
    local objectStorage, storageArea = self:getCurrentSelection()

    self.currentConfigurationIndex = 1

    if objectStorage.sortedAcceptedTypes ~= nil and storageArea and storageArea.objectType ~= nil then
        local sortedAcceptedTypes = storageArea.sortedAcceptedTypes or objectStorage.sortedAcceptedTypes

        for i = 1, #sortedAcceptedTypes do
            if sortedAcceptedTypes[i].fillTypeIndex == storageArea.fillTypeIndex then
                if sortedAcceptedTypes[i].objectTypeIndex == storageArea.objectType.index then
                    self.currentConfigurationIndex = i
                    self.configurationConfirmButtonInfo.disabled = true

                    break
                end
            end
        end
    end

    self.configurationActive = true
    self:updateInfoTexts(true)

    self.configurationList:reloadData()
    self.configurationList:setSelectedIndex(self.currentConfigurationIndex)

    self:updateMenuButtons()
    self:updateVisibleFrames()

    FocusManager:setFocus(self.configurationList)
end

function InGameMenuObjectStorageFrame:onButtonConfigurationBack()
    self.updateTime = g_currentMission.time + 5000

    self.configurationActive = false
    self.currentConfigurationIndex = nil
    self.configurationConfirmButtonInfo.disabled = false

    self.objectStorageList:reloadData()
    self:updateVisibleFrames()

    FocusManager:setFocus(self.objectStorageList)
end

function InGameMenuObjectStorageFrame:onButtonConfigurationConfirm()
    if self.configurationConfirmButtonInfo.disabled then
        -- self:playSample(GuiSoundPlayer.SOUND_SAMPLES.ERROR)

        return false
    end

    local objectStorage, storageArea = self:getCurrentSelection()
    local configurationText = "shop_messageConfigurationChangeFailed"

    if storageArea ~= nil then
        local acceptedType = objectStorage:getAcceptedType(self.configurationList:getSelectedIndexInSection(), storageArea.sortedAcceptedTypes)

        if acceptedType ~= nil and objectStorage:setStorageObjectType(storageArea.index, acceptedType.objectTypeIndex, acceptedType.fillTypeIndex, false) then
            configurationText = "shop_messageConfigurationChanged"
        end
    end

    self:onButtonConfigurationBack()

    self:addInfoText(self.i18n:getText(configurationText), InGameMenuObjectStorageFrame.INFO_TEXT_SPAWNER, true)

    return true
end

function InGameMenuObjectStorageFrame:onDoubleClickConfiguration(index)
    if self:onButtonConfigurationConfirm() then
        self:playSample(GuiSoundPlayer.SOUND_SAMPLES.CLICK)
    end
end

function InGameMenuObjectStorageFrame:addInfoText(text, slot, raiseUpdate, updateTime)
    local infoText = self.infoTexts[slot]

    if infoText ~= nil then
        infoText.text = text or ""
        infoText.updateTime = g_currentMission.time + (updateTime or 10000)

        if raiseUpdate then
            self:updateInfoTexts(false)
        end
    end
end

function InGameMenuObjectStorageFrame:updateInfoTexts(clear)
    local text, numTexts = "", 0

    self.hasInfoText = false

    for i, infoText in ipairs (self.infoTexts) do
        if not clear then
            if infoText.text ~= "" then
                numTexts = numTexts + 1

                if numTexts == 1 then
                    text = string.format("- %s", infoText.text)
                else
                    text = string.format("%s\n\n- %s", text, infoText.text)
                end
            end
        else
            infoText.text = ""
            infoText.updateTime = 0
        end
    end

    self.infoBoxText:setText(text)
    self.hasInfoText = numTexts > 0
end
