--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 25.08.2019
Revision: FS22-03

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

local modDirectory = g_currentModDirectory
local modName = g_currentModName

local versionString = "0.0.0.0"
local buildId = 5

local loadConsoleCommands = StartParams.getIsSet("consoleCommandsGtX")
local overwrittenFunctions = {}

local objectStorageManager
local validationFail

local placeableTypesInitialised = false
local vehicleTypesInitialised = false

local customModNames = {
    "FS22_MaizePlus"
}

local function isActive()
    return g_modIsLoaded[modName] and objectStorageManager ~= nil
end

local function overwrittenFunction(object, funcName, newFunc)
    if object ~= nil then
        local originalFunc = object[funcName]

        if originalFunc ~= nil then
            object[funcName] = function (self, ...)
                return newFunc(self, originalFunc, ...)
            end

            table.insert(overwrittenFunctions, {
                object = object,
                funcName = funcName,
                originalFunc = originalFunc
            })
        end
    end
end

local function getPageIndex(inGameMenu, list)
    if inGameMenu.pageProduction ~= nil then
        for i, frame in ipairs (list) do
            if frame == inGameMenu.pageProduction then
                return i + 1
            end
        end
    end

    return 12
end

local function updatePagingElement(inGameMenu)
    local index = getPageIndex(inGameMenu, inGameMenu.pagingElement.elements)
    local pageObjectStorage = inGameMenu.pageObjectStorage

    for i, frame in ipairs (inGameMenu.pagingElement.elements) do
        if frame == pageObjectStorage then
            table.remove(inGameMenu.pagingElement.elements, i)
            table.insert(inGameMenu.pagingElement.elements, index, frame)

            break
        end
    end

    for i, frame in ipairs (inGameMenu.pagingElement.pages) do
        if frame.element == pageObjectStorage then
            table.remove(inGameMenu.pagingElement.pages, i)
            table.insert(inGameMenu.pagingElement.pages, index, frame)

            break
        end
    end
end

local function updatePageFrames(inGameMenu)
    local index = getPageIndex(inGameMenu, inGameMenu.pageFrames)
    local pageObjectStorage = inGameMenu.pageObjectStorage

    for i, frame in ipairs (inGameMenu.pageFrames) do
        if frame == pageObjectStorage then
            table.remove(inGameMenu.pageFrames, i)
            table.insert(inGameMenu.pageFrames, index, frame)

            break
        end
    end

    inGameMenu:rebuildTabList()
end

local function unload(mission)
    if isActive() then
        objectStorageManager:unloadMapData()
    end

    if g_globalMods ~= nil then
        g_globalMods.objectStorageManager = nil
    end

    if mission ~= nil then
        mission.objectStorageManager = nil
    end

    g_objectStorageManager = nil
    objectStorageManager = nil

    for i = #overwrittenFunctions, 1, -1 do
        local data = overwrittenFunctions[i]

        if data.object ~= nil and data.originalFunc ~= nil then
            data.object[data.funcName] = data.originalFunc
        end

        overwrittenFunctions[i] = nil
    end
end

local function loadMapData(baleManager, superFunc, xmlFile, missionInfo, ...)
    local success = superFunc(baleManager, xmlFile, missionInfo, ...)

    if success and isActive() then
        local customBaleEnvironments = {}

        local function loadBaleHudOverlaysFromXML(modFile, modDir, customEnv)
            local modDesc = XMLFile.loadIfExists("ModDesc", modFile)

            if modDesc ~= nil then
                if modDesc:hasProperty("modDesc.objectStorage.baleHudOverlays.hudOverlay(0)") then
                    modDesc:iterate("modDesc.objectStorage.baleHudOverlays.hudOverlay", function (index, key)
                        local fillTypeName = modDesc:getString(key .. "#fillType")
                        local filename = modDesc:getString(key .. "#filename")

                        if fillTypeName ~= nil and filename ~= nil then
                            local isRoundbale = modDesc:getBool(key .. "#isRoundbale", false)

                            objectStorageManager:addHudOverlayFilename(isRoundbale, fillTypeName, filename, modDir, customEnv)
                        end
                    end)
                end

                modDesc:delete()
            end
        end

        -- First add the base icons to be used and a backup for round and square
        loadBaleHudOverlaysFromXML(modDirectory .. "modDesc.xml", modDirectory, modName)

        -- Check if mod map adds bale icons for a new bale fillType
        if missionInfo.customEnvironment ~= nil then
            local mod = g_modManager:getModByName(missionInfo.customEnvironment)

            if mod ~= nil then
                loadBaleHudOverlaysFromXML(mod.modFile, mod.modDir, mod.modName)
            end

            customBaleEnvironments[missionInfo.customEnvironment] = true
        end

        -- If a mod adds bales outside of the default fill types then also check for bale icons
        for index, bale in ipairs(baleManager.bales) do
            if bale.customEnvironment ~= nil and not customBaleEnvironments[bale.customEnvironment] then
                local mod = g_modManager:getModByName(bale.customEnvironment)

                if mod ~= nil then
                    loadBaleHudOverlaysFromXML(mod.modFile, mod.modDir, mod.modName)
                end

                customBaleEnvironments[bale.customEnvironment] = true
            end
        end

        -- Allow some extra mods to add custom icons as these do not add bales from the modDesc
        for _, customModName in ipairs (customModNames) do
            if g_modIsLoaded[customModName] and not customBaleEnvironments[customModName] then
                local mod = g_modManager:getModByName(customModName)

                if mod ~= nil then
                    loadBaleHudOverlaysFromXML(mod.modFile, mod.modDir, mod.modName)
                end

                customBaleEnvironments[customModName] = true
            end
        end

        objectStorageManager:loadMapData(xmlFile, missionInfo, ...)
    end

    return success
end

local function loadMapFinished(mission)
    if isActive() then
        if objectStorageManager.inGameMenuFrame ~= nil then
            local inGameMenu = g_gui.screenControllers[InGameMenu]

            updatePagingElement(inGameMenu)
            updatePageFrames(inGameMenu)
        end

        mission.objectStorageManager = objectStorageManager
        objectStorageManager:loadMapFinished(mission)

        -- No entity exists check carried out in base game code when pushing objects, done here to avoid delivery errors
        -- NOTE: Fixed in game update v1.8.1.0 :-)

       -- local oldThrowObject = Player.throwObject

        -- function Player.throwObject(self, noEventSend)
            -- if self.pickedUpObject ~= nil and self.pickedUpObjectJointId ~= nil then
                -- if not entityExists(self.pickedUpObject) then
                    -- self:pickUpObject(false)

                    -- return
                -- end
            -- elseif self.isObjectInRange and self.lastFoundObject ~= nil and not entityExists(self.lastFoundObject) then
                -- return
            -- end

            -- oldThrowObject(self, noEventSend)
        -- end
    end
end

local function validateTypes(typeManager)
    if typeManager.typeName == "placeable" then
        if not placeableTypesInitialised then
            BaleManager.loadMapData = Utils.overwrittenFunction(BaleManager.loadMapData, loadMapData)
            FSBaseMission.loadMapFinished = Utils.prependedFunction(FSBaseMission.loadMapFinished, loadMapFinished)
        end

        if g_iconGenerator == nil and isActive() then
            local inGameMenu = g_gui.screenControllers[InGameMenu]

            if inGameMenu ~= nil then
                local xmlFile = loadXMLFile("ingameMenuFrameReferenceXML", modDirectory .. "gui/IngameMenuObjectStorageFrameRef.xml")

                if xmlFile ~= nil and xmlFile ~= 0 then
                    -- Clear any existing controls (reload etc)
                    inGameMenu.controlIDs["pageObjectStorage"] = nil

                    -- Register controls
                    inGameMenu:registerControls({"pageObjectStorage"})

                    -- Add new element
                    g_gui:loadGuiRec(xmlFile, "GuiElements", inGameMenu.pagingElement, inGameMenu)

                    -- Expose new controls
                    inGameMenu:exposeControlsAsFields("pageObjectStorage")

                    -- Sort the 'pagingElement'
                    updatePagingElement(inGameMenu)

                    -- Update mapping and position data
                    inGameMenu.pagingElement:updateAbsolutePosition()
                    inGameMenu.pagingElement:updatePageMapping()

                    delete(xmlFile)
                end

                local objectStorageFrame = g_gui:resolveFrameReference(inGameMenu.pageObjectStorage)

                -- Store GUI overwrite and restore it on delete
                overwrittenFunction(InGameMenu, "setPlayerFarm", function (self, superFunc, farm)
                    if self.pageObjectStorage ~= nil then
                        self.pageObjectStorage:setPlayerFarm(farm)
                    end

                    superFunc(self, farm)
                end)

                inGameMenu:registerPage(inGameMenu.pageObjectStorage, nil, function()
                    if inGameMenu.missionDynamicInfo == nil or inGameMenu.playerFarmId == nil then
                        return true
                    end

                    return not inGameMenu.missionDynamicInfo.isMultiplayer or inGameMenu.playerFarmId ~= FarmManager.SPECTATOR_FARM_ID
                end)

                inGameMenu:addPageTab(inGameMenu.pageObjectStorage, g_iconsUIFilename, GuiUtils.getUVs({195, 65, 65, 65}))

                inGameMenu.pageObjectStorage:applyScreenAlignment()
                inGameMenu.pageObjectStorage:updateAbsolutePosition()
                inGameMenu.pageObjectStorage:onGuiSetupFinished()

                inGameMenu.messageCenter:subscribe("OPEN_OBJECT_STORAGE_SCREEN", function (self, objectStorage)
                    if objectStorage ~= nil and self.pageObjectStorage ~= nil then
                        self:changeScreen(InGameMenu)

                        local index = self.pagingElement:getPageMappingIndexByElement(self.pageObjectStorage)

                        self.pageSelector:setState(index, true)
                        self.pageObjectStorage:setSelectedObjectStorage(objectStorage)
                    end
                end, inGameMenu)

                -- Sort the 'pageFrames'
                updatePageFrames(inGameMenu)

                -- Initialise frame
                objectStorageFrame:initialize()

                -- Add direct access for the manager
                objectStorageManager.inGameMenuFrame = objectStorageFrame
                objectStorageManager.cameraDialog = cameraDialog
            end
        end

        placeableTypesInitialised = true
    elseif typeManager.typeName == "vehicle" then
        -- There is now so many mods adding and changing pallets even just for sell points for some strange reason along with unrealistic capacities.
        -- So I need a way to set and sync the capacity if the original pallet added to storage no longer exists and the available one is too small.
        if not vehicleTypesInitialised then
            FillUnit.onPostLoad = Utils.prependedFunction(FillUnit.onPostLoad, function(self, savegame)
                if self.isServer and savegame ~= nil then
                    local capacity = savegame.xmlFile:getFloat(savegame.key .. ".fillUnit#objectStorageCapacity")

                    if capacity ~= nil and capacity > 0 then
                        local fillUnit = self.spec_fillUnit.fillUnits[1]

                        if fillUnit ~= nil and fillUnit.capacity < capacity then
                            fillUnit.capacity = capacity
                            self.objectStorageCapacity = capacity
                        end
                    end
                end
            end)

            FillUnit.saveToXMLFile = Utils.prependedFunction(FillUnit.saveToXMLFile, function(self, xmlFile, key, usedModNames)
                if self.objectStorageCapacity ~= nil then
                    xmlFile:setFloat(key .. "#objectStorageCapacity", self.objectStorageCapacity)
                end
            end)

            FillUnit.onWriteStream = Utils.prependedFunction(FillUnit.onWriteStream, function(self, streamId, connection)
                if not connection:getIsServer() then
                    if streamWriteBool(streamId, self.objectStorageCapacity ~= nil) then
                        streamWriteFloat32(streamId, self.objectStorageCapacity)
                    end
                end
            end)

            FillUnit.onReadStream = Utils.prependedFunction(FillUnit.onReadStream, function(self, streamId, connection)
                if connection:getIsServer() then
                    if streamReadBool(streamId) then
                        self:setFillUnitCapacity(1, streamReadFloat32(streamId), true)
                    end
                end
            end)
        end

        vehicleTypesInitialised = true
    end
end

local function validateMod()
    if g_globalMods ~= nil then
        local mod = g_modManager:getModByName(modName)

        if g_globalMods.objectStorageManager ~= nil then
            Logging.warning("[Object Storage] Validation of '%s' failed, already loaded by '%s'.", mod.modName, g_globalMods.objectStorageManager.customEnvironment)

            return false
        end

        versionString = mod.version or versionString

        -- Name cannot change or other mods dependent on 'FS22_ObjectStorage' will not load.
        -- Support '_update' for internal update testing only as this will also break dependent mods.
        if mod.modName == "FS22_ObjectStorage" or mod.modName == "FS22_ObjectStorage_update" then
            if mod.author ~= nil and #mod.author == 3 then
                return true
            end
        end

        validationFail = {
            startUpdateTime = 2000,

            update = function(self, dt)
                if g_currentMission == nil or not g_currentMission.gameStarted then
                    return
                end

                self.startUpdateTime = self.startUpdateTime - dt

                if self.startUpdateTime < 0 then
                    local text = string.format(g_i18n:getText("ui_objectStorage_loadError", mod.modName), mod.modName, mod.author or "Unknown")

                    if g_dedicatedServerInfo == nil then
                        if not g_gui:getIsGuiVisible() then
                            -- Swapped button texts so 'ESC' now ignores the message to avoid accidental clicking.
                            g_gui:showYesNoDialog({
                                title = string.format("%s - Version %s", mod.title, versionString),
                                text = text,
                                dialogType = DialogElement.TYPE_LOADING,
                                callback = self.openModHubLink,
                                yesText = g_i18n:getText("button_modHubDownload"),
                                noText = g_i18n:getText("button_dismiss")
                            })
                        end
                    else
                        print("\n" .. text .. "\n    - https://farming-simulator.com/mods.php?lang=en&country=be&title=fs2022&filter=org&org_id=129652&page=0" .. "\n")
                        self.openModHubLink(false)
                    end
                end
            end,

            openModHubLink = function(yes)
                if yes then
                    local language = g_languageShort
                    local link = "mods.php?lang=en&country=be&title=fs2022&filter=org&org_id=129652&page=0"

                    if language == "de" or language == "fr" then
                        link = "mods.php?lang=" .. language .. "&country=be&title=fs2022&filter=org&org_id=129652&page=0"
                    end

                    openWebFile(link, "")
                end

                removeModEventListener(validationFail)
                validationFail = nil
            end
        }

        addModEventListener(validationFail)
    else
        Logging.warning("[Object Storage] Validation of '%s' failed due to base code changes!", modName)
    end

    return false
end

local function init()
    -- GUI Frame
    source(modDirectory .. "scripts/gui/InGameMenuObjectStorageFrame.lua")
    source(modDirectory .. "scripts/gui/CameraDialog.lua")

    -- Manager / Miscellaneous
    source(modDirectory .. "scripts/misc/ObjectStorageManager.lua")
    source(modDirectory .. "scripts/misc/ObjectStorageSpawner.lua")
    source(modDirectory .. "scripts/misc/ObjectStorageVisibilityNodes.lua")

    -- Object Storage
    source(modDirectory .. "scripts/objects/ObjectStorage.lua")
    source(modDirectory .. "scripts/objects/BaleObjectStorage.lua")
    source(modDirectory .. "scripts/objects/PalletObjectStorage.lua")

    -- Events
    source(modDirectory .. "scripts/events/ObjectStorageWarningMessageEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageSpawnObjectsEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageSpawnerStatusEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageAddBaleObjectEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageAddObjectEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageRemoveObjectEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageSetObjectTypeEvent.lua")
    source(modDirectory .. "scripts/events/ObjectStorageSetInputTriggerEvent.lua")

    if g_iconGenerator == nil and g_gui ~= nil then
        g_gui:loadProfiles(modDirectory .. "gui/guiProfiles.xml")

        if validateMod() then
            local objectStorageFrame = InGameMenuObjectStorageFrame.new(g_messageCenter, g_i18n)
            local cameraDialog = CameraDialog.new()

            g_gui:loadGui(modDirectory .. "gui/InGameMenuObjectStorageFrame.xml", "ObjectStorageFrame", objectStorageFrame, true)
            g_gui:loadGui(modDirectory .. "gui/CameraDialog.xml", "CameraDialog", cameraDialog)

            objectStorageManager = ObjectStorageManager.new(modDirectory, modName)

            objectStorageManager.developmentMode = buildId < 1
            objectStorageManager.loadConsoleCommands = loadConsoleCommands
            objectStorageManager.version = versionString
            objectStorageManager.build = buildId

            g_globalMods.objectStorageManager = objectStorageManager
            g_objectStorageManager = objectStorageManager

            FSBaseMission.delete = Utils.appendedFunction(FSBaseMission.delete, unload)
            TypeManager.validateTypes = Utils.prependedFunction(TypeManager.validateTypes, validateTypes)
        end
    end
end

init()
