--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 25.08.2019
Revision: FS22-03

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

ObjectStorageManager = {}

ObjectStorageManager.MOD_NAME = g_currentModName
ObjectStorageManager.MOD_DIR = g_currentModDirectory

ObjectStorageManager.FERMENTING_FACTOR = 24 * 60 * 60 * 1000 -- days / months
ObjectStorageManager.DEFAULT_WRAP_COLOUR = {1, 1, 1, 1}

ObjectStorageManager.DEFAULT_PALLET_SIZE = {
    heightOffset = 0,
    lengthOffset = 0,
    widthOffset = 0,
    width = 1.6,
    length = 1.6,
    height = 1.2
}

ObjectStorageManager.MAX_STORAGES = 60 -- Not used on PC at this stage

local ObjectStorageManager_mt = Class(ObjectStorageManager)
local EMPTY_TABLE = {}

function ObjectStorageManager.new(modDirectory, modName, customMt)
    local self = setmetatable({}, customMt or ObjectStorageManager_mt)

    self.isServer = g_server ~= nil
    self.isClient = g_client ~= nil

    self.baseDirectory = modDirectory
    self.customEnvironment = modName

    self.bales = {}
    self.packedBales = {}
    self.baleFillTypes = {}
    self.baleFermentedFillTypes = {}

    self.baleObjectTypes = {}
    self.individualBaleObjectTypes = {}

    self.palletFillTypes = {}
    self.palletObjectTypes = {}
    self.individualPalletObjectTypes = {}
    self.palletFilenameToSize = {}

    self.objectStorages = {}
    self.objectStoragesByFarm = {}

    self.fermentingStorages = {}
    self.indexedFermentingStorages = {}

    self.roundBaleHudOverlayFilenames = {}
    self.squareBaleHudOverlayFilenames = {}

    self.debugSpawnAreas = false
    self.debugSpawnAreaInfo = nil

    return self
end

function ObjectStorageManager:loadMapData(xmlFile, missionInfo, baseDirectory)
    self:loadBaleTypes(baseDirectory)
    self:loadPalletTypes(baseDirectory)

    return true
end

function ObjectStorageManager:loadMapFinished()
    g_currentMission:addUpdateable(self)

    addConsoleCommand("gtxObjectStorageSortFermentingBales", "Sorts the bale order for all fermenting storage buildings. This is for SP debug and testing, use with caution [fermentingFirst] [triggerOnly]", "consoleCommandSortFermentingBales", self)
    addConsoleCommand("gtxObjectStorageGetTotalFillLevels", "Returns total fill levels for given farm id [farmId] [ignoreEmpty]", "consoleCommandGetTotalFillLevels", self)
    addConsoleCommand("gtxObjectStorageDebugSpawnAreas", "Toggles spawn area debug mode", "consoleCommandDebugSpawnAreas", self)

    if self.developmentMode and self.isServer then
        addConsoleCommand("gtxObjectStorageSetOwner", "Set current storage farm id", "consoleCommandDebugSetOwner", self)
    end
end

function ObjectStorageManager:unloadMapData()
    g_currentMission:removeUpdateable(self)

    removeConsoleCommand("gtxObjectStorageSortFermentingBales")
    removeConsoleCommand("gtxObjectStorageGetTotalFillLevels")
    removeConsoleCommand("gtxObjectStorageDebugSpawnAreas")

    if self.developmentMode and self.isServer then
        removeConsoleCommand("gtxObjectStorageSetOwner")
    end
end

function ObjectStorageManager:delete()
end

function ObjectStorageManager:update(dt)
    if self.isServer then
        local numFermentingStorages = #self.indexedFermentingStorages

        if numFermentingStorages > 0 then
            local effectiveTimeScale = g_currentMission:getEffectiveTimeScale()

            for i = numFermentingStorages, 1, -1 do
                local storage = self.indexedFermentingStorages[i]

                if storage.isDeleting or not storage:updateFermentation(dt, effectiveTimeScale) then
                    table.remove(self.indexedFermentingStorages, i)
                    self.fermentingStorages[storage] = nil
                end
            end
        end
    end
end

function ObjectStorageManager:addObjectStorage(storage)
    if self.objectStorages[storage] then
        return false
    end

    self.objectStorages[storage] = true

    local farmId = storage:getOwnerFarmId()

    if farmId ~= AccessHandler.EVERYONE then
        if self.objectStoragesByFarm[farmId] == nil then
            self.objectStoragesByFarm[farmId] = {}
        end

        if table.addElement(self.objectStoragesByFarm[farmId], storage) then
            self:sortFarmStorages(farmId)

            storage:onObjectStorageAdded()
        end

        if self.inGameMenuFrame ~= nil and self.inGameMenuFrame.frameOpen then
            self.inGameMenuFrame:reloadStorageListData()
        end

        return true
    end

    return false
end

function ObjectStorageManager:removeObjectStorage(storage)
    if storage == nil then
        return false
    end

    local farmId = storage:getOwnerFarmId()

    self:removeFermentingStorage(storage)

    self.objectStorages[storage] = nil

    if farmId ~= AccessHandler.EVERYONE then
        local storagesByFarm = self.objectStoragesByFarm[farmId]

        if storagesByFarm ~= nil then
            for i = 1, #storagesByFarm do
                if storagesByFarm[i] == storage then
                    table.remove(storagesByFarm, i)

                    break
                end
            end

            if #storagesByFarm > 0 then
                self:sortFarmStorages(farmId)
            else
                self.objectStoragesByFarm[farmId] = nil
            end

            if self.inGameMenuFrame ~= nil and self.inGameMenuFrame.frameOpen then
                self.inGameMenuFrame:reloadStorageListData()
            end
        end
    end

    return true
end

function ObjectStorageManager:addFermentingStorage(storage)
    if self.fermentingStorages[storage] == nil then
        self.fermentingStorages[storage] = storage

        table.addElement(self.indexedFermentingStorages, storage)
    end
end

function ObjectStorageManager:removeFermentingStorage(storage)
    table.removeElement(self.indexedFermentingStorages, storage)

    self.fermentingStorages[storage] = nil
end

function ObjectStorageManager:getHudOverlayFilename(fillTypeIndex, isBale, isRoundbale)
    fillTypeIndex = fillTypeIndex or FillType.UNKNOWN

    if isBale then
        local hudOverlays = self.squareBaleHudOverlayFilenames

        if isRoundbale then
            hudOverlays = self.roundBaleHudOverlayFilenames
        end

        if hudOverlays[fillTypeIndex] ~= nil then
            return hudOverlays[fillTypeIndex]
        end

        return hudOverlays[FillType.UNKNOWN]
    end

    local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex)

    if fillType == nil then
        fillType = g_fillTypeManager:getFillTypeByIndex(FillType.UNKNOWN)
    end

    return fillType.hudOverlayFilename
end

function ObjectStorageManager:sortFarmStorages(farmId)
    if self.objectStoragesByFarm[farmId] == nil then
        return
    end

    table.sort(self.objectStoragesByFarm[farmId], function (a, b)
        if a.baleStorage == b.baleStorage then
            return a:getName() < b:getName()
        end

        if a.baleStorage then
            return true
        end

        return false
    end)
end

function ObjectStorageManager:getStorages()
    return self.objectStorages or EMPTY_TABLE
end

function ObjectStorageManager:getFarmStorages(farmId)
    return self.objectStoragesByFarm[farmId] or EMPTY_TABLE
end

function ObjectStorageManager:getCanAddStorage()
    return true
end

function ObjectStorageManager:getMaxStorages()
    return ObjectStorageManager.MAX_STORAGES
end

function ObjectStorageManager:getTotalFillLevelsByFarmId(farmId)
    local fillLevels = {}

    -- If no farmId is given then use the players farmId
    if farmId == nil then
        farmId = g_currentMission:getFarmId()
    end

    for _, storage in ipairs (self:getFarmStorages(farmId)) do
        for _, storageArea in ipairs (storage.indexedStorageAreas) do
            local objectFillLevel = fillLevels[storageArea.fillTypeIndex] or 0

            for _, object in ipairs (storageArea.objects) do
                objectFillLevel = objectFillLevel + (object.fillLevel or 0)
            end

            fillLevels[storageArea.fillTypeIndex] = objectFillLevel
        end
    end

    return fillLevels -- Returns a table of fill levels keyed by fill type index
end

function ObjectStorageManager:loadBaleTypes(baseDirectory)
    local function getIsFilePathInvalid(path)
        return path == nil or not fileExists(path)
    end

    -- Need more complete fillTypes information so build my own bales list, also try and avoid mods incorrectly making changes
    for index, bale in ipairs(g_baleManager.bales) do
        local xmlFile = XMLFile.load("temporaryBale", bale.xmlFilename, BaleManager.baleXMLSchema)

        if xmlFile ~= nil then
            if bale.isAvailable then
                local baleInfo = {
                    xmlFilename = bale.xmlFilename,
                    isRoundbale = bale.isRoundbale,
                    baseDirectory = bale.baseDirectory,
                    customEnvironment = bale.customEnvironment,
                    width = bale.width,
                    height = bale.height,
                    length = bale.length,
                    diameter = bale.diameter,
                    baleIndex = index,
                    fillTypes = {}
                }

                -- Fix for mods such as MaizePlus that add bales without 'customEnvironment' or 'baseDirectory'
                local baleModName, baleBaseDirectory = Utils.getModNameAndBaseDirectory(bale.xmlFilename)

                if baleModName ~= nil then
                    baleInfo.baseDirectory = baleBaseDirectory
                    baleInfo.customEnvironment = baleModName
                end

                Bale.loadFillTypesFromXML(baleInfo.fillTypes, xmlFile, baleInfo.baseDirectory or baseDirectory)

                if getIsFilePathInvalid(baleInfo.fillTypes.diffuseFilename) then
                    baleInfo.fillTypes.diffuseFilename = nil
                end

                if getIsFilePathInvalid(baleInfo.fillTypes.normalFilename) then
                    baleInfo.fillTypes.normalFilename = nil
                end

                if getIsFilePathInvalid(baleInfo.fillTypes.specularFilename) then
                    baleInfo.fillTypes.specularFilename = nil
                end

                if getIsFilePathInvalid(baleInfo.fillTypes.alphaFilename) then
                    baleInfo.fillTypes.alphaFilename = nil
                end

                baleInfo.fillTypes.forceAcceleration = nil
                baleInfo.fillTypes.mass = nil

                ObjectStorageManager.addBaleObjectType(self, baleInfo)

                table.insert(self.bales, baleInfo)
            end

            if xmlFile:getValue("bale.packedBale#singleBale") ~= nil then
                self.packedBales[bale.xmlFilename] = bale
            end

            xmlFile:delete()
        end
    end

    table.sort(self.baleObjectTypes, function (a, b)
        if a.isRoundbale == b.isRoundbale then
            if a.isRoundbale then
                if a.diameter == b.diameter then
                    return a.width < b.width
                end

                return a.diameter < b.diameter
            end

            return a.length < b.length
        end

        if a.isRoundbale then
            return true
        end

        return false
    end)

    for i, objectType in ipairs (self.baleObjectTypes) do
        objectType.index = i

        for fillTypeIndex, _ in pairs (objectType.fillTypeIndexs) do
            table.insert(self.individualBaleObjectTypes, {
                fillTypeIndex = fillTypeIndex,
                objectType = objectType,
                objectTypeIndex = i
            })
        end
    end
end

function ObjectStorageManager:getSortedAcceptedBaleTypes(acceptedFillTypes, minDiameter, maxDiameter, minLength, maxLength)
    local acceptedTypes = {}

    if acceptedFillTypes ~= nil then
        minDiameter = minDiameter or 0.1
        maxDiameter = maxDiameter or math.huge
        minLength = minLength or 0.1
        maxLength = maxLength or math.huge

        for _, individualType in ipairs (self.individualBaleObjectTypes) do
            if acceptedFillTypes[individualType.fillTypeIndex] then
                if ObjectStorageManager.getBaleSizeInRange(individualType.objectType, minDiameter, maxDiameter, minLength, maxLength) then
                    table.insert(acceptedTypes, individualType)
                end
            end
        end

        table.sort(acceptedTypes, function (a, b)
            local nameA = g_fillTypeManager:getFillTypeNameByIndex(a.fillTypeIndex):lower()
            local nameB = g_fillTypeManager:getFillTypeNameByIndex(b.fillTypeIndex):lower()

            if nameA == nameB then
                local objectTypeA = a.objectType
                local objectTypeB = b.objectType

                if objectTypeA.isRoundbale == objectTypeB.isRoundbale then
                    if objectTypeA.isRoundbale then
                        if objectTypeA.diameter == objectTypeB.diameter then
                            return objectTypeA.width < objectTypeB.width
                        end

                        return objectTypeA.diameter < objectTypeB.diameter
                    end

                    return objectTypeA.length < objectTypeB.length
                end

                if objectTypeA.isRoundbale then
                    return true
                end

                return false
            end

            return nameA < nameB
        end)
    end

    return acceptedTypes
end

function ObjectStorageManager:getAvailableBaleObjectTypes()
    return self.baleObjectTypes
end

function ObjectStorageManager:getAvailableBaleFillTypes()
    return self.baleFillTypes
end

function ObjectStorageManager:getBaleAttributesByObjectType(objectType, fillTypeIndex, farmId)
    if objectType ~= nil and fillTypeIndex ~= nil then
        for _, bale in ipairs(self.bales) do
            if ObjectStorageManager.getBaleMatchesSize(bale, objectType.width, objectType.height, objectType.length, objectType.diameter) then
                for i = 1, #bale.fillTypes do
                    local fillTypeInfo = bale.fillTypes[i]

                    if fillTypeInfo.fillTypeIndex == fillTypeIndex then
                        local attributes = {
                            fillType = fillTypeIndex,
                            xmlFilename = bale.xmlFilename,
                            capacity = fillTypeInfo.capacity,
                            fillLevel = fillTypeInfo.capacity,
                            supportsWrapping = fillTypeInfo.supportsWrapping,
                            wrappingState = 0,
                            baleValueScale = 1,
                            isMissionBale = false,
                            isFermenting = false
                        }

                        attributes.farmId = farmId or AccessHandler.EVERYONE
                        attributes.wrappingColor = ObjectStorageManager.DEFAULT_WRAP_COLOUR

                        return attributes, bale
                    end
                end
            end
        end
    end

    return nil
end

function ObjectStorageManager:getBaleAttributesByFilename(xmlFilename, fillTypeIndex, farmId)
    if xmlFilename ~= nil and fillTypeIndex ~= nil then
        for _, bale in ipairs(self.bales) do
            if bale.xmlFilename == xmlFilename then
                for i = 1, #bale.fillTypes do
                    local fillTypeInfo = bale.fillTypes[i]

                    if fillTypeInfo.fillTypeIndex == fillTypeIndex then
                        local attributes = {
                            fillType = fillTypeIndex,
                            xmlFilename = bale.xmlFilename,
                            capacity = fillTypeInfo.capacity,
                            fillLevel = fillTypeInfo.capacity,
                            supportsWrapping = fillTypeInfo.supportsWrapping,
                            wrappingState = 0,
                            baleValueScale = 1,
                            isMissionBale = false,
                            isFermenting = false
                        }

                        attributes.farmId = farmId or AccessHandler.EVERYONE
                        attributes.wrappingColor = ObjectStorageManager.DEFAULT_WRAP_COLOUR

                        return attributes, bale
                    end
                end
            end
        end
    end

    return nil
end

function ObjectStorageManager:getBaleFillTypeInfoByObjectType(objectType, fillTypeIndex)
    for _, bale in pairs (self.bales) do
        if bale.isRoundbale == objectType.isRoundbale then
            local fillTypeInfo

            for i = 1, #bale.fillTypes do
                if bale.fillTypes[i].fillTypeIndex == fillTypeIndex then
                    fillTypeInfo = bale.fillTypes[i]

                    break
                end
            end

            if fillTypeInfo ~= nil and ObjectStorageManager.getBaleMatchesSize(bale, objectType.width, objectType.height, objectType.length, objectType.diameter) then
                return fillTypeInfo
            end
        end
    end

    return nil
end

function ObjectStorageManager:getBaleSizeByObjectTypeIndex(objectTypeIndex)
    local objectType = self.baleObjectTypes[objectTypeIndex]

    if objectType ~= nil then
        if objectType.isRoundbale then
            return (objectType.diameter or 0) * 100
        end

        return (objectType.length or 0) * 100
    end

    return 0
end

function ObjectStorageManager:getBaleClassByFilename(xmlFilename)
    if self.packedBales ~= nil and self.packedBales[xmlFilename] then
        return PackedBale
    end

    return Bale
end

function ObjectStorageManager:getIsFermentedFillType(fillTypeIndex)
    if fillTypeIndex == FillType.SILAGE then
        return true
    end

    return self.baleFermentedFillTypes[fillTypeIndex]
end

function ObjectStorageManager:addHudOverlayFilename(isRoundbale, fillTypeName, filename, baseDirectory, customEnvironment)
    local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)
    local hudOverlayFilename = Utils.getFilename(filename, baseDirectory)

    if fillTypeIndex ~= nil and hudOverlayFilename ~= nil and fileExists(hudOverlayFilename) then
        if isRoundbale then
            if self.roundBaleHudOverlayFilenames[fillTypeIndex] == nil or customEnvironment == ObjectStorageManager.MOD_NAME then
                self.roundBaleHudOverlayFilenames[fillTypeIndex] = hudOverlayFilename
            end
        else
            if self.squareBaleHudOverlayFilenames[fillTypeIndex] == nil or customEnvironment == ObjectStorageManager.MOD_NAME then
                self.squareBaleHudOverlayFilenames[fillTypeIndex] = hudOverlayFilename
            end
        end
    end
end

function ObjectStorageManager.addBaleObjectType(self, bale)
    for i = 1, #self.baleObjectTypes do
        local existingBale = self.baleObjectTypes[i]

        if existingBale.isRoundbale == bale.isRoundbale and ObjectStorageManager.getBaleMatchesSize(existingBale, bale.width, bale.height, bale.length, bale.diameter) then
            for _, fillTypeData in pairs (bale.fillTypes) do
                existingBale.fillTypeIndexs[fillTypeData.fillTypeIndex] = true

                self.baleFillTypes[fillTypeData.fillTypeIndex] = true

                if fillTypeData.fermenting ~= nil and fillTypeData.fermenting.outputFillTypeIndex ~= nil then
                    self.baleFermentedFillTypes[fillTypeData.fermenting.outputFillTypeIndex] = true
                end
            end

            return
        end
    end

    local fillTypeIndexs = {}

    for _, fillTypeData in pairs (bale.fillTypes) do
        fillTypeIndexs[fillTypeData.fillTypeIndex] = true

        self.baleFillTypes[fillTypeData.fillTypeIndex] = true

        if fillTypeData.fermenting ~= nil and fillTypeData.fermenting.outputFillTypeIndex ~= nil then
            self.baleFermentedFillTypes[fillTypeData.fermenting.outputFillTypeIndex] = true
        end
    end

    table.insert(self.baleObjectTypes, {
        unitShort = "l",
        isPallet = false,
        width = bale.width,
        height = bale.height,
        length = bale.length,
        diameter = bale.diameter,
        isRoundbale = bale.isRoundbale,
        fillTypeIndexs = fillTypeIndexs
    })
end

function ObjectStorageManager.getBaleMatchesSize(bale, width, height, length, diameter)
    if bale.isRoundbale then
        return diameter == bale.diameter and width == bale.width
    else
        return width == bale.width and height == bale.height and length == bale.length
    end
end

function ObjectStorageManager.getBaleSizeInRange(sizeData, minDiameter, maxDiameter, minLength, maxLength)
    if sizeData.isRoundbale then
        return sizeData.diameter >= minDiameter and sizeData.diameter <= maxDiameter
    else
        return sizeData.length >= minLength and sizeData.length <= maxLength
    end

    return false
end

function ObjectStorageManager:loadPalletTypes(baseDirectory)
    local palletFilenames = {}

    local function addPalletFilename(fillTypeName, palletFilename, baseDir)
        local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

        if fillTypeIndex ~= nil then
            palletFilename = Utils.getFilename(palletFilename, baseDir)

            if g_storeManager:getItemByXMLFilename(palletFilename) ~= nil then
                palletFilenames[fillTypeIndex] = palletFilename
            end
        end
    end

    addPalletFilename("TREESAPLINGS", "$data/objects/pallets/treeSaplingPallet/treeSaplingPallet.xml")
    addPalletFilename("WOODCHIPS", "$data/objects/pallets/fillablePallet/fillablePallet.xml")
    addPalletFilename("POPLAR", "$data/objects/pallets/palletPoplar/palletPoplar.xml")
    addPalletFilename("ROADSALT", "$data/objects/bigBagPallet/roadSalt/bigBagPallet_roadSalt.xml")
    addPalletFilename("PIGFOOD", "$data/objects/bigBagPallet/pigFood/bigBagPallet_pigFood.xml")
    addPalletFilename("WHEAT", "$data/objects/bigBagPallet/chickenFood/bigBagPallet_chickenFood.xml")
    addPalletFilename("OAT", "$data/objects/bigBagPallet/horseFood/bigBagPallet_horseFood.xml")

    if FillType.LIQUIDSEEDTREATMENT ~= nil then
        local agiPackDirectory = g_modNameToDirectory["pdlc_agiPack"]

        if agiPackDirectory ~= nil then
            addPalletFilename("LIQUIDSEEDTREATMENT", "objects/pallets/liquidSeedTreatmentPallet/liquidSeedTreatmentPallet.xml", agiPackDirectory)
        end
    end

    if FillType.MAPLESYRUP ~= nil then
        local fillType = g_fillTypeManager:getFillTypeByIndex(FillType.MAPLESYRUP)
        local hudOverlayFilename = ObjectStorageManager.MOD_DIR .. "shared/hud/hud_fill_mapleSyrup.dds"

        if fillType ~= nil then
            local mapleSyrupProductionDirectory = g_modNameToDirectory["FS22_MapleSyrupProduction"]

            -- Default to my pallet, it is a production so most likely to be stored
            if mapleSyrupProductionDirectory ~= nil then
                addPalletFilename("MAPLESYRUP", "pallets/mapleSyrupPallet.xml", mapleSyrupProductionDirectory)
            end

            if fileExists(hudOverlayFilename) then
                fillType.hudOverlayFilename = hudOverlayFilename
            end
        end
    end

    -- Listed as litres in base game so fixed here, there is no short unit i.e (pcs) so use full unit name.
    local piecesShort = g_i18n:getText("unit_pieces")
    local litersShort = g_i18n:getText("unit_literShort")

    local fillTypeToUnitShort = {
        [FillType.TREESAPLINGS] = piecesShort,
        [FillType.EGG] = piecesShort
    }

    -- Only load pallets registered as part of fillTypes.xml or my defaults
    -- Pallets can be added by any mod or map from the modDesc and fillTypes.xml
    for fillTypeIndex, fillType in ipairs(g_fillTypeManager.fillTypes) do
        local palletFilename = palletFilenames[fillTypeIndex] or fillType.palletFilename

        if palletFilename ~= nil then
            local createPallet = true

            for i = 1, #self.palletObjectTypes do
                local existingPallet = self.palletObjectTypes[i]

                if existingPallet.xmlFilename == palletFilename then
                    existingPallet.fillTypeIndexs[fillTypeIndex] = true
                    createPallet = false

                    break
                end
            end

            if createPallet then
                local size = StoreItemUtil.getSizeValues(palletFilename, "vehicle", 0, {})

                local palletObjectType = {
                    xmlFilename = palletFilename,
                    isPallet = true,
                    width = size.width,
                    height = size.height,
                    length = size.length
                }

                palletObjectType.fillTypeIndexs = {
                    [fillTypeIndex] = true
                }

                local palletXmlFile = XMLFile.load("palletXmlFilename", palletFilename, Vehicle.xmlSchema)
                local rootName = palletXmlFile:getRootName()
                local capacity, unitTextOverride = 0, fillTypeToUnitShort[fillTypeIndex]

                palletXmlFile:iterate(rootName .. ".fillUnit.fillUnitConfigurations.fillUnitConfiguration", function (_, key)
                    palletXmlFile:iterate(key .. ".fillUnits.fillUnit", function (_, fillUnitKey)
                        capacity = math.max(capacity, palletXmlFile:getValue(fillUnitKey .. "#capacity") or 0)

                        if unitTextOverride == nil then
                            unitTextOverride = palletXmlFile:getValue(fillUnitKey .. "#unitTextOverride")

                            if unitTextOverride ~= nil and unitTextOverride ~= "" then
                                if unitTextOverride:sub(1, 6) == "$l10n_" then
                                    unitTextOverride = g_i18n:getText(unitTextOverride:sub(7))
                                end
                            else
                                unitTextOverride = nil
                            end
                        end
                    end)
                end)

                local unitShort = unitTextOverride or fillType.unitShort

                if unitShort == nil or unitShort == "" then
                    unitShort = litersShort
                end

                palletObjectType.capacity = capacity
                palletObjectType.unitShort = unitShort

                table.insert(self.palletObjectTypes, palletObjectType)

                palletXmlFile:delete()
            end

            self.palletFillTypes[fillTypeIndex] = true
        end
    end

    -- table.sort(self.palletObjectTypes, function (a, b)
        -- return a.xmlFilename < b.xmlFilename
    -- end)

    for i, objectType in ipairs (self.palletObjectTypes) do
        objectType.index = i

        for fillTypeIndex, _ in pairs (objectType.fillTypeIndexs) do
            table.insert(self.individualPalletObjectTypes, {
                fillTypeIndex = fillTypeIndex,
                objectType = objectType,
                objectTypeIndex = i
            })
        end
    end
end

function ObjectStorageManager:getSortedAcceptedPalletTypes(acceptedFillTypes)
    local acceptedTypes = {}

    if acceptedFillTypes ~= nil then
        for _, individualType in ipairs (self.individualPalletObjectTypes) do
            if acceptedFillTypes[individualType.fillTypeIndex] then
                table.insert(acceptedTypes, individualType)
            end
        end

        table.sort(acceptedTypes, function (a, b)
            local nameA = g_fillTypeManager:getFillTypeNameByIndex(a.fillTypeIndex):lower()
            local nameB = g_fillTypeManager:getFillTypeNameByIndex(b.fillTypeIndex):lower()

            return nameA < nameB
        end)
    end

    return acceptedTypes
end

function ObjectStorageManager:getPalletAttributesByFillType(fillTypeIndex)
    if fillTypeIndex ~= nil then
        for _, pallet in ipairs(self.palletObjectTypes) do
            for fillType, _ in pairs (pallet.fillTypeIndexs) do
                if fillType == fillTypeIndex then
                    local attributes = {
                        fillType = fillType,
                        xmlFilename = pallet.xmlFilename,
                        capacity = pallet.capacity,
                        fillLevel = pallet.capacity
                    }

                    return attributes, pallet
                end
            end
        end
    end

    return nil
end

function ObjectStorageManager:getPalletAttributesByFilename(xmlFilename, fillTypeIndex)
    if xmlFilename ~= nil and fillTypeIndex ~= nil then
        for _, pallet in ipairs(self.palletObjectTypes) do
            if xmlFilename == pallet.xmlFilename then
                for fillType, _ in pairs (pallet.fillTypeIndexs) do
                    if fillType == fillTypeIndex then
                        local attributes = {
                            fillType = fillType,
                            xmlFilename = pallet.xmlFilename,
                            capacity = pallet.capacity,
                            fillLevel = pallet.capacity
                        }

                        return attributes, pallet
                    end
                end
            end
        end
    end

    return nil
end

function ObjectStorageManager:getAvailablePalletObjectTypes()
    return self.palletObjectTypes
end

function ObjectStorageManager:getAvailablePalletFillTypes()
    return self.palletFillTypes
end

function ObjectStorageManager:getPalletSizeByFilename(xmlFilename)
    if xmlFilename ~= nil then
        local size = self.palletFilenameToSize[xmlFilename]

        if size == nil then
            local xmlFile = XMLFile.load("storeItemSizeXml", xmlFilename, Vehicle.xmlSchema)

            if xmlFile ~= nil then
                size = StoreItemUtil.getSizeValuesFromXMLByKey(xmlFile, "vehicle", "base", "size", "size", 0, EMPTY_TABLE, ObjectStorageManager.DEFAULT_PALLET_SIZE)

                xmlFile:delete()
            else
                size = ObjectStorageManager.DEFAULT_PALLET_SIZE
            end

            self.palletFilenameToSize[xmlFilename] = size
        end

        return size
    end

    return ObjectStorageManager.DEFAULT_PALLET_SIZE
end

function ObjectStorageManager.getFilename(filename, baseDirectory)
    if filename ~= nil then
        if filename:sub(1, 15) == "$objectStorage$" then
            return ObjectStorageManager.MOD_DIR .. filename:sub(16), false
        end

        return Utils.getFilename(filename, baseDirectory)
    end

    return nil
end

function ObjectStorageManager:draw()
    if self.debugSpawnAreas and self.debugSpawnAreaInfo ~= nil then
        local info = self.debugSpawnAreaInfo

        for storage, _ in pairs (self.objectStorages) do
            if storage.spawnArea ~= nil then
                if info.area and info.area > 0 then
                    local storageArea = storage.indexedStorageAreas[info.area] or storage.indexedStorageAreas[1]

                    if storageArea ~= nil then
                        local objectType = storageArea.objectType
                        local isRoundbale = Utils.getNoNil(objectType.isRoundbale, false)
                        local height = isRoundbale and objectType.diameter or objectType.height
                        local length = isRoundbale and objectType.diameter or objectType.length

                        storage.spawnArea:drawAreas(objectType.width, height, length, storage.baleStorage, isRoundbale, storageArea)
                    end
                else
                    storage.spawnArea:drawAreas(info.sizeX, info.sizeY, info.sizeZ, info.isBale, info.isRoundbale)
                end
            end
        end
    end
end

function ObjectStorageManager:consoleCommandSortFermentingBales(fermentingFirst, triggerOnly)
    if g_currentMission == nil then
        return "Command only available when game started!"
    end

    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return "Command only available in single player for debug and testing purposes!"
    end

    local numAreasSorted = 0
    local numStoragesSorted = 0

    fermentingFirst = Utils.stringToBoolean(fermentingFirst)
    triggerOnly = Utils.stringToBoolean(triggerOnly)

    local function sortFermentingStorage(a, b)
        if a.isFermenting and b.isFermenting then
            if fermentingFirst then
                return (a.fermentationTime or 0) > (b.fermentationTime or 0)
            end

            return (a.fermentationTime or 0) < (b.fermentationTime or 0)
        end

        return (a.isFermenting and not fermentingFirst) or (b.isFermenting and fermentingFirst)
    end

    local function canSortFermentingStorage(storage)
        if storage == nil then
            return false, false
        end

        if triggerOnly and storage.activatable ~= nil then
            return g_currentMission.activatableObjectsSystem.objects[storage.activatable] ~= nil, true
        end

        return true, false
    end

    for _, storage in ipairs (self.indexedFermentingStorages) do
        local canSortStorage, isTargetStorage = canSortFermentingStorage(storage)

        if canSortStorage then
            for _, storageArea in ipairs (storage.indexedStorageAreas) do
                if storageArea.supportsFermenting then
                    storage:updateVisibilityNodes(storageArea, 0, storageArea.maxObjects)

                    table.sort(storageArea.objects, sortFermentingStorage)

                    storageArea.numObjects = #storageArea.objects
                    storage:updateVisibilityNodes(storageArea, storageArea.numObjects, storageArea.maxObjects)

                    numAreasSorted = numAreasSorted + 1
                end
            end

            if g_gui:getIsGuiVisible() then
                g_messageCenter:publish("OBJECT_STORAGE_UPDATE_GUI", storage)
            end

            if isTargetStorage then
                return string.format("Sorted %d fermenting storage area(s) in storage %s.", numAreasSorted, storage:getName() or "")
            end

            numStoragesSorted = numStoragesSorted + 1
        elseif isTargetStorage then
            return "No storage trigger found or in range. Enter a valid fermenting storage trigger or set parameter 2 [triggerOnly] to 'false' to sort all."
        end
    end

    return string.format("Sorted %d fermenting storage area(s) across %d storage(s).", numAreasSorted, numStoragesSorted)
end

function ObjectStorageManager:consoleCommandDebugSpawnAreas(area, sizeX, sizeY, sizeZ, isBale, isRoundbale)
    area = tonumber(area or 0) -- If 'area' is greater than 0 then the current configuration information will be used

    if area > 0 then
        if self.debugSpawnAreas and self.debugSpawnAreaInfo ~= nil then
            if self.debugSpawnAreaInfo.area ~= area then
                self.debugSpawnAreaInfo.area = area

                return "Updating displayed spawn area index on " .. tostring(table.size(self.objectStorages)) .. " storage(s)"
            end
        end
    else
        if self.debugSpawnAreas and (sizeX == nil or sizeY == nil or sizeZ == nil) then
            g_currentMission:removeDrawable(self)

            self.debugSpawnAreas = false
            self.debugSpawnAreaInfo = nil

            return "Spawn area size display disabled"
        end

        sizeX = tonumber(sizeX or 2)
        sizeY = tonumber(sizeY or 2)
        sizeZ = tonumber(sizeZ or 2)

        isBale = Utils.stringToBoolean(isBale)
        isRoundbale = Utils.stringToBoolean(isRoundbale)

        if self.debugSpawnAreas and self.debugSpawnAreaInfo ~= nil then
            local info = self.debugSpawnAreaInfo

            if info.sizeX ~= sizeX or info.sizeY ~= sizeY or info.sizeZ ~= sizeZ or info.isBale ~= isBale or info.isRoundbale ~= isRoundbale then
                info.area = 0
                info.sizeX = sizeX
                info.sizeY = sizeY
                info.sizeZ = sizeZ

                info.isBale = isBale
                info.isRoundbale = isRoundbale

                return "Updating displayed spawn size values on " .. tostring(table.size(self.objectStorages)) .. " storage building(s)"
            end
        end
    end

    self.debugSpawnAreas = not self.debugSpawnAreas

    if self.debugSpawnAreas then
        self.debugSpawnAreaInfo = {
            area = area,
            sizeX = sizeX,
            sizeY = sizeY,
            sizeZ = sizeZ,
            isBale = isBale,
            isRoundbale = isRoundbale
        }

        g_currentMission:addDrawable(self)

        return "Spawn area size display enabled on " .. tostring(table.size(self.objectStorages)) .. " storage building(s)"
    end

    g_currentMission:removeDrawable(self)

    self.debugSpawnAreaInfo = nil

    return "Spawn area size display disabled"
end

function ObjectStorageManager:consoleCommandGetTotalFillLevels(farmId, ignoreEmpty)
    if g_currentMission == nil then
        return "Command only available when game started!"
    end

    local farm = g_farmManager:getFarmById(tonumber(farmId) or g_currentMission:getFarmId())

    if farm == nil or farm.farmId == 0 then
        return "Invalid farm id given!"
    end

    local fillLevelsInfo = {}
    local litreUnit = g_i18n:getText("unit_liter")
    local displayEmpty = not Utils.stringToBoolean(ignoreEmpty)

    for fillTypeIndex, fillLevel in pairs (self:getTotalFillLevelsByFarmId(farm.farmId)) do
        if displayEmpty or fillLevel > 0 then
            table.insert(fillLevelsInfo, {
                title = g_fillTypeManager:getFillTypeTitleByIndex(fillTypeIndex),
                fillLevel = fillLevel
            })
        end
    end

    if #fillLevelsInfo == 0 then
        return string.format("Farm %s (%d) does not have any valid Object Storage buildings.", farm.name, farm.farmId)
    end

    table.sort(fillLevelsInfo, function(a, b)
        return a.title < b.title
    end)

    local text = string.format("\nTotal fill levels for %s (%d) Object Storage buildings:\n", farm.name, farm.farmId)

    for _, info in ipairs (fillLevelsInfo) do
        text = string.format("%s  - %s: %s\n", text, info.title, g_i18n:formatVolume(info.fillLevel, 0, litreUnit))
    end

    return text
end

function ObjectStorageManager:consoleCommandDebugSetOwner(farmId)
    local objectStorage = nil
    local lastDistance = 25

    local x, y, z = getWorldTranslation(g_currentMission.player.cameraNode)

    for storage, _ in pairs (self.objectStorages) do
        local tx, _, tz = getWorldTranslation(storage.interactionTrigger or storage.node)
        local distance = MathUtil.getPointPointDistance(tx, tz, x, z)

        if distance <= lastDistance then
            distance = lastDistance
            objectStorage = storage
        end
    end

    if objectStorage ~= nil and objectStorage.owningPlaceable ~= nil then
        if farmId == nil or tonumber(farmId) == nil then
            farmId = g_currentMission:getFarmId()
        end

        objectStorage.owningPlaceable:setOwnerFarmId(tonumber(farmId), true)

        return string.format("Storage '%s' farm id changed to %d", objectStorage:getName(), farmId)
    end

    return "No storage withing 25 meters"
end
