--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 02.01.2022
Revision: FS22-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

ObjectStorageVisibilityNodes = {}

ObjectStorageVisibilityNodes.xmlSchema = nil

ObjectStorageVisibilityNodes.TYPE_MATCHED = 0
ObjectStorageVisibilityNodes.TYPE_CLOSEST_MATCH = 1
ObjectStorageVisibilityNodes.TYPE_BACKUP = 2
ObjectStorageVisibilityNodes.TYPE_NONE = 3

local ObjectStorageVisibilityNodes_mt = Class(ObjectStorageVisibilityNodes)

local function cloneVisibilityNode(node, doClone)
    if doClone then
        return clone(node, false, false, true)
    end

    return node
end

local function debugLog(enabled, ...)
    if enabled then
        local text = ""

        for i = 1, select("#", ...) do
            text = text .. " " .. tostring(select(i, ...))
        end

        print(text)
    end
end

g_xmlManager:addCreateSchemaFunction(function ()
    ObjectStorageVisibilityNodes.xmlSchema = XMLSchema.new("ObjectStorageVisibilityNodes")
end)

g_xmlManager:addInitSchemaFunction(function ()
    local schema = ObjectStorageVisibilityNodes.xmlSchema

    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?)#filename", "filename")
    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?)#node", "node id")
    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?)#name", "identifier")

    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).hideByIndex#node", "Path to node using a shader with 'mergeChildren_hideByIndex'.")
    schema:register(XMLValueType.INT, "visibilityNodes.visibilityNode(?).hideByIndex#numMeshes", "Number of indexed meshes", 0)

    schema:register(XMLValueType.BOOL, "visibilityNodes.visibilityNode(?).baleMeshes#applyAlpha", "Apply alpha to mesh", true)
    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).baleMeshes.baleMesh(?)#node", "Path to mesh node")
    schema:register(XMLValueType.BOOL, "visibilityNodes.visibilityNode(?).baleMeshes.baleMesh(?)#supportsWrapping", "Defines if the mesh is hidden while wrapping or not", false)
    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).baleMeshes.baleMesh(?)#ignoreFillTypes", "Fill types that should hide the mesh")

    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).objects.object(?)#node", "Object visibility node")
    schema:register(XMLValueType.BOOL, "visibilityNodes.visibilityNode(?).objects#treeSaplingPallet", "Enable random rotation and scale of object nodes", false)

    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).fillTypeMaterials.material(?)#fillType", "Fill type name")
    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).fillTypeMaterials.material(?)#node", "Node which receives material")
    schema:register(XMLValueType.STRING, "visibilityNodes.visibilityNode(?).fillTypeMaterials.material(?)#refNode", "Node which provides material")
end)

function ObjectStorageVisibilityNodes.new(customMt)
    local self = setmetatable({}, customMt or ObjectStorageVisibilityNodes_mt)

    self.owner = nil
    self.storageArea = nil
    self.sharedLoadRequestIds = nil

    self.visibilityNodes = {}
    self.updateableNodes = false
    self.hideByShaderMeshes = 0

    self.typeId = ObjectStorageVisibilityNodes.TYPE_NONE
    self.debugPrintsEnabled = false

    return self
end

function ObjectStorageVisibilityNodes:load(storageArea, owner)
    if owner == nil or storageArea == nil or storageArea.fillTypeIndex == nil or storageArea.objectType == nil then
        return false
    end

    local node = storageArea.visibilityNodesLinkNode or owner.node
    local variation = storageArea.visibilityNodesVariation or 1
    local sharedVisibilityNodes = owner.sharedVisibilityNodes

    self.debugPrintsEnabled = Utils.getNoNil(sharedVisibilityNodes.debugPrintsEnabled, false)

    self.owner = owner
    self.storageArea = storageArea

    local sharedData, typeId = self:getSharedVisibilityNode(sharedVisibilityNodes, storageArea.fillTypeIndex, storageArea.objectType, owner.palletStorage, variation)

    self.typeId = typeId
    self.variation = variation

    if sharedData ~= nil and node ~= nil then
        local linkNodesRootNode, linkNodesSharedLoadRequestId = g_i3DManager:loadSharedI3DFile(sharedVisibilityNodes.linkNodesFilename, false, false)

        if linkNodesRootNode == 0 then
            Logging.warning("File '%s' does not exist or failed to load!", tostring(sharedVisibilityNodes.linkNodesFilename))

            return false
        end

        self.sharedLoadRequestIds = {
            linkNodesSharedLoadRequestId
        }

        local linkNodes = I3DUtil.indexToObject(linkNodesRootNode, sharedData.linkNodes)
        local numLinkNodes = linkNodes ~= nil and getNumOfChildren(linkNodes) or 0
        local maxObjects = math.min(2 ^ ObjectStorage.OBJECTS_SEND_NUM_BITS - 1, 650)

        if numLinkNodes == 0 or numLinkNodes > maxObjects then
            delete(linkNodesRootNode)

            Logging.warning("No link nodes found in I3D '%s' or number of nodes exceeds maximum of '%d' at node '%s'!", sharedVisibilityNodes.linkNodesFilename, maxObjects, tostring(sharedData.linkNodes))

            return false
        end

        self.linkNodes = linkNodes
        self.name = sharedData.name
        self.fillTypeIndex = storageArea.fillTypeIndex

        link(node, linkNodes)
        delete(linkNodesRootNode)

        local key = ""
        local xmlFile = XMLFile.loadIfExists("sharedVisibilityNodes", sharedData.xmlFilename, ObjectStorageVisibilityNodes.xmlSchema)

        if xmlFile == nil then
            Logging.warning("File '%s' does not exist or failed to load!", tostring(sharedData.xmlFilename))

            return false
        end

        xmlFile:iterate("visibilityNodes.visibilityNode", function (_, visibilityNodeKey)
            if xmlFile:getValue(visibilityNodeKey .. "#name") == sharedData.name then
                key = visibilityNodeKey

                return false
            end
        end)

        if key == nil then
            Logging.xmlWarning(xmlFile, "A key with 'visibilityNodes.visibilityNode(?)#name' %s could not be found!", tostring(sharedData.name))

            xmlFile:delete()

            return false
        end

        local i3dFilename = ObjectStorageManager.getFilename(xmlFile:getValue(key .. "#filename"), self.owner.baseDirectory)

        if i3dFilename == nil or not fileExists(i3dFilename) then
            xmlFile:delete()

            return false
        end

        local rootNode, sharedLoadRequestId = g_i3DManager:loadSharedI3DFile(i3dFilename, false, false)

        if rootNode == 0 then
            xmlFile:delete()

            return false
        end

        table.insert(self.sharedLoadRequestIds, sharedLoadRequestId)

        local baseNode = I3DUtil.indexToObject(rootNode, xmlFile:getValue(key .. "#node", "0"))

        if baseNode == nil then
            delete(rootNode)
            xmlFile:delete()

            return false
        end

        if owner.baleStorage then
            self:loadBaleVisibilityNodes(xmlFile, key, baseNode, numLinkNodes, linkNodes, storageArea.fillTypeIndex, storageArea.objectType, sharedData.isBackup)
        else
            self:loadPalletVisibilityNodes(xmlFile, key, baseNode, numLinkNodes, linkNodes, storageArea.fillTypeIndex)
        end

        delete(rootNode)
        xmlFile:delete()

        return true
    end

    self.owner = nil
    self.storageArea = nil

    return false
end

function ObjectStorageVisibilityNodes:delete()
    if self.sharedLoadRequestIds ~= nil then
        for i = 1, #self.sharedLoadRequestIds do
            g_i3DManager:releaseSharedI3DFile(self.sharedLoadRequestIds[i])
        end

        self.sharedLoadRequestIds = nil
    end

    self.visibilityNodes = {}
    self.updateableNodes = false
    self.hideByShaderMeshes = 0

    if self.linkNodes ~= nil then
        delete(self.linkNodes)
        self.linkNodes = nil
    end
end

function ObjectStorageVisibilityNodes:onFillLevelChanged(objects, fillLevel, capacity)
    local numVisibilityNodes = #self.visibilityNodes

    if numVisibilityNodes > 0 then
        local updateableNodes = self.updateableNodes
        local visibleNodes = fillLevel
        local isEqual = true

        if capacity > 0 and numVisibilityNodes ~= capacity then
            visibleNodes = math.ceil(numVisibilityNodes * (fillLevel / capacity))
            isEqual = false
        end

        for i = 1, numVisibilityNodes do
            local visibilityNode = self.visibilityNodes[i]
            local active = i <= visibleNodes

            if active ~= visibilityNode.active then
                if active then
                    if isEqual and updateableNodes then
                        local object = objects ~= nil and objects[i]

                        if object ~= nil then
                            if visibilityNode.meshes then
                                ObjectStorageVisibilityNodes.setVisibilityNodeMeshes(visibilityNode, object)
                            elseif visibilityNode.objectNodes then
                                local numObjectNodes = #visibilityNode.objectNodes
                                local visibleIndex = numObjectNodes

                                if object.fillLevel ~= nil and object.capacity ~= nil and object.capacity > 0 then
                                    local fillLevelPct = object.fillLevel / object.capacity

                                    visibleIndex = math.floor(numObjectNodes * fillLevelPct)

                                    if visibleIndex == 0 and fillLevelPct > 0 then
                                        visibleIndex = 1
                                    end
                                end

                                for j = 1, numObjectNodes do
                                    setVisibility(visibilityNode.objectNodes[j], j <= visibleIndex)
                                end
                            elseif visibilityNode.hideByIndexNode then
                                local hiddenIndexs = 0

                                if object.fillLevel ~= nil and object.capacity ~= nil and object.capacity > 0 then
                                    local fillLevelPct = object.fillLevel / object.capacity

                                    hiddenIndexs = self.hideByShaderMeshes - math.floor(self.hideByShaderMeshes * fillLevelPct)

                                    if hiddenIndexs >= self.hideByShaderMeshes and fillLevelPct > 0 then
                                        hiddenIndexs = self.hideByShaderMeshes - 1
                                    end
                                end

                                setShaderParameter(visibilityNode.hideByIndexNode, "hideByIndex", hiddenIndexs, 0, 0, 0, false)
                            end
                        end
                    end

                    setVisibility(visibilityNode.node, true)
                    setRigidBodyType(visibilityNode.node, RigidBodyType.STATIC)
                else
                    setVisibility(visibilityNode.node, false)
                    setRigidBodyType(visibilityNode.node, RigidBodyType.NONE)
                end

                visibilityNode.active = active
            end
        end
    end
end

function ObjectStorageVisibilityNodes:setPlacementActive()
    for _, visibilityNode in ipairs (self.visibilityNodes) do
        setVisibility(visibilityNode.node, true)
        visibilityNode.active = nil
    end
end

function ObjectStorageVisibilityNodes:loadBaleVisibilityNodes(xmlFile, key, baseNode, numLinkNodes, linkNodes, fillTypeIndex, objectType, isBackup)
    local baseMeshes = {}

    xmlFile:iterate(key .. ".baleMeshes.baleMesh", function (_, meshKey)
        local nodeStr = xmlFile:getValue(meshKey .. "#node")

        if I3DUtil.indexToObject(baseNode, nodeStr) ~= nil then
            local baseMesh = {
                supportsWrapping = xmlFile:getValue(meshKey .. "#supportsWrapping", true),
                nodeStr = nodeStr
            }

            local ignoreFillTypes = xmlFile:getValue(meshKey .. "#ignoreFillTypes")

            if ignoreFillTypes ~= nil then
                local fillTypeNames = string.split(ignoreFillTypes, " ")

                for _, fillTypeName in ipairs (fillTypeNames) do
                    local fillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

                    if fillType == fillTypeIndex then
                        baseMesh.ignoreFillType = true
                    end
                end
            end

            table.insert(baseMeshes, baseMesh)
        end
    end)

    if not isBackup and #baseMeshes > 0 then
        local fillTypeInfo = g_objectStorageManager:getBaleFillTypeInfoByObjectType(objectType, fillTypeIndex)

        if fillTypeInfo ~= nil then
            local applyAlpha = xmlFile:getValue(key .. ".baleMeshes#applyAlpha", true)

            ObjectStorageVisibilityNodes.setFillTypeTexturesForNode(baseNode, fillTypeInfo, applyAlpha)
        end
    end

    setRigidBodyType(baseNode, RigidBodyType.NONE)
    setVisibility(baseNode, false)

    local numBaseMeshes = #baseMeshes

    for i = 0, numLinkNodes - 1 do
        local node = cloneVisibilityNode(baseNode, i > 0)
        local meshes

        link(getChildAt(linkNodes, i), node)

        if numBaseMeshes > 0 then
            meshes = {}

            for j = 1, numBaseMeshes do
                local baseMesh = baseMeshes[j]
                local meshNode = I3DUtil.indexToObject(node, baseMesh.nodeStr)

                if meshNode ~= nil then
                    table.insert(meshes, {
                        ignoreFillType = baseMesh.ignoreFillType,
                        supportsWrapping = baseMesh.supportsWrapping,
                        node = meshNode
                    })
                end
            end

            if meshes[1] == nil then
                meshes = nil
            end
        end

        table.insert(self.visibilityNodes, {
            meshes = meshes,
            active = false,
            node = node
        })
    end

    self.updateableNodes = numBaseMeshes > 0
end

function ObjectStorageVisibilityNodes:loadPalletVisibilityNodes(xmlFile, key, baseNode, numLinkNodes, linkNodes, fillTypeIndex)
    local baseObjectNodes = {}
    local hideByIndex = nil
    local treeSaplingPallet = false

    if xmlFile:hasProperty(key .. ".objects") then
        treeSaplingPallet = xmlFile:getValue(key .. ".objects#treeSaplingPallet", false)

        xmlFile:iterate(key .. ".objects.object", function (_, objectKey)
            local nodeStr = xmlFile:getValue(objectKey .. "#node")

            if I3DUtil.indexToObject(baseNode, nodeStr) ~= nil then
                table.insert(baseObjectNodes, nodeStr)
            end
        end)
    elseif xmlFile:hasProperty(key .. ".hideByIndex") then
        local nodeStr = xmlFile:getValue(key .. ".hideByIndex#node")
        local node = I3DUtil.indexToObject(baseNode, nodeStr)
        local numMeshes = xmlFile:getValue(key .. ".hideByIndex#numMeshes", 0)

        if numMeshes > 0 and node ~= nil and getHasShaderParameter(node, "hideByIndex") then
            hideByIndex = nodeStr
            self.hideByShaderMeshes = numMeshes
        end
    end

    if xmlFile:hasProperty(key .. ".fillTypeMaterials") then
        xmlFile:iterate(key .. ".fillTypeMaterials.material", function (_, materialKey)
            local fillTypeName = xmlFile:getValue(materialKey .. "#fillType")

            if fillTypeName ~= nil then
                local fillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

                if fillType ~= nil and fillType == fillTypeIndex then
                    local node = I3DUtil.indexToObject(baseNode, xmlFile:getValue(materialKey .. "#node"))
                    local refNode = I3DUtil.indexToObject(baseNode, xmlFile:getValue(materialKey .. "#refNode"))

                    if node ~= nil and refNode ~= nil then
                        setMaterial(node, getMaterial(refNode, 0), 0)
                    else
                        Logging.xmlWarning(xmlFile, "Missing node or ref node in '%s'", materialKey)
                    end
                end
            else
                Logging.xmlWarning(xmlFile, "Missing fill type in '%s'", materialKey)
            end
        end)
    end

    setRigidBodyType(baseNode, RigidBodyType.NONE)
    setVisibility(baseNode, false)

    local numBaseObjectNodes = #baseObjectNodes

    for i = 0, numLinkNodes - 1 do
        local node = cloneVisibilityNode(baseNode, i > 0)
        local objectNodes, hideByIndexNode = nil, nil

        link(getChildAt(linkNodes, i), node)

        if numBaseObjectNodes > 0 then
            objectNodes = {}

            for j = 1, numBaseObjectNodes do
                local objectNode = I3DUtil.indexToObject(node, baseObjectNodes[j])

                if objectNode ~= nil then
                    -- If it is a sapling pallet then randomise the saplings
                    if treeSaplingPallet then
                        setRotation(objectNode, 0, math.random(0, math.pi * 2), 0)
                        setScale(objectNode, 1, math.random(90, 110) / 100, 1)
                    end

                    table.insert(objectNodes, objectNode)
                end
            end

            if objectNodes[1] == nil then
                objectNodes = nil
            end
        elseif hideByIndex ~= nil then
            hideByIndexNode = I3DUtil.indexToObject(node, hideByIndex)
        end

        table.insert(self.visibilityNodes, {
            objectNodes = objectNodes,
            hideByIndexNode = hideByIndexNode,
            active = false,
            node = node
        })
    end

    if numBaseObjectNodes > 0 or hideByIndex ~= nil then
        self.updateableNodes = true
    end
end

function ObjectStorageVisibilityNodes:getSharedVisibilityNode(sharedData, fillTypeIndex, objectType, palletStorage, variation)
    if sharedData ~= nil and sharedData.availableTypes ~= nil then
        local width = objectType.width
        local height = objectType.height
        local length = objectType.length
        local diameter = objectType.diameter
        local isRoundbale = objectType.isRoundbale

        for i = 1, #sharedData.availableTypes do
            local availableType = sharedData.availableTypes[i]

            if availableType.variation == variation and availableType.fillTypes[fillTypeIndex] then
                if palletStorage then
                    debugLog(self.debugPrintsEnabled, "Pallet Matched: ", availableType.name)

                    return availableType, ObjectStorageVisibilityNodes.TYPE_MATCHED
                end

                if isRoundbale == availableType.isRoundbale then
                    if isRoundbale then
                        if diameter == availableType.diameter and width == availableType.width then
                            debugLog(self.debugPrintsEnabled, "Bale Matched: ", availableType.name)

                            return availableType, ObjectStorageVisibilityNodes.TYPE_MATCHED
                        end
                    else
                        if width == availableType.width and height == availableType.height and length == availableType.length then
                            debugLog(self.debugPrintsEnabled, "Bale Matched: ", availableType.name)

                            return availableType, ObjectStorageVisibilityNodes.TYPE_MATCHED
                        end
                    end
                end
            end
        end

        if not palletStorage then
            local closestType = nil
            local lastDistance = math.huge
            local distance = lastDistance

            for i = 1, #sharedData.availableTypes do
                local availableType = sharedData.availableTypes[i]

                if availableType.variation == variation and availableType.fillTypes[fillTypeIndex] and isRoundbale == availableType.isRoundbale then
                    if isRoundbale then
                        distance = MathUtil.vector2Length(width - availableType.width, diameter - availableType.diameter)
                    else
                        distance = MathUtil.vector3Length(width - availableType.width, height - availableType.height, length - availableType.length)
                    end

                    if distance < lastDistance then
                        lastDistance = distance
                        closestType = availableType
                    end
                end
            end

            if closestType ~= nil and lastDistance <= 1 then
                debugLog(self.debugPrintsEnabled, "Closest Match: ", closestType.name)

                return closestType, ObjectStorageVisibilityNodes.TYPE_CLOSEST_MATCH
            end
        end

        for i = 1, #sharedData.availableTypes do
            local availableType = sharedData.availableTypes[i]

            if availableType.variation == variation and availableType.isBackup then
                if palletStorage or isRoundbale == availableType.isRoundbale then
                    debugLog(self.debugPrintsEnabled, "Backup Match: ", availableType.name)

                    return availableType, ObjectStorageVisibilityNodes.TYPE_BACKUP
                end
            end
        end

        if variation > 1 and self.owner ~= nil and self.storageArea ~= nil then
            Logging.warning("[%s] Failed to set visibility nodes for storage area %d, no compatible shared visibility nodes using variation %d found!", self.owner:getName(), self.storageArea.index, variation)
        end
    end

    return nil, ObjectStorageVisibilityNodes.TYPE_NONE
end

function ObjectStorageVisibilityNodes:getNumVisibilityNodes()
    return #self.visibilityNodes
end

function ObjectStorageVisibilityNodes.setFillTypeTexturesForNode(node, fillTypeInfo, applyAlpha)
    local numChildren = getNumOfChildren(node)

    for i = 1, numChildren do
        ObjectStorageVisibilityNodes.setFillTypeTexturesForNode(getChildAt(node, i - 1), fillTypeInfo, applyAlpha)
    end

    if getHasClassId(node, ClassIds.SHAPE) then
        local materialId = getMaterial(node, 0)

        if materialId ~= 0 then
            local shaderFilename = getMaterialCustomShaderFilename(materialId)

            if shaderFilename:contains("silageBaleShader") then
                local oldWrappingState, _, _, _ = getShaderParameter(node, "wrappingState")
                local oldWrapR, oldWrapG, oldWrapB, oldWrapA = getShaderParameter(node, "colorScale")

                if fillTypeInfo.diffuseFilename ~= nil then
                    materialId = setMaterialDiffuseMapFromFile(materialId, fillTypeInfo.diffuseFilename, true, true, false)
                end

                if fillTypeInfo.normalFilename ~= nil then
                    materialId = setMaterialNormalMapFromFile(materialId, fillTypeInfo.normalFilename, true, false, false)
                end

                if fillTypeInfo.specularFilename ~= nil then
                    materialId = setMaterialGlossMapFromFile(materialId, fillTypeInfo.specularFilename, true, true, false)
                end

                if applyAlpha and fillTypeInfo.alphaFilename ~= nil then
                    materialId = setMaterialCustomMapFromFile(materialId, "alphaMap", fillTypeInfo.alphaFilename, true, false, false)
                end

                setMaterial(node, materialId, 0)
                setShaderParameter(node, "wrappingState", oldWrappingState, 0, 0, 0, false)
                setShaderParameter(node, "colorScale", oldWrapR, oldWrapG, oldWrapB, oldWrapA, false)
            end
        end
    end
end

function ObjectStorageVisibilityNodes.setVisibilityNodeMeshes(visibilityNode, attributes)
    local wrappingState = attributes.wrappingState or 0

    local r, g, b, a = 1, 1, 1, 1
    local isWrapped = wrappingState > 0

    if isWrapped and attributes.wrappingColor ~= nil and #attributes.wrappingColor == 4 then
        r, g, b, a = unpack(attributes.wrappingColor)
    end

    for i = 1, #visibilityNode.meshes do
        local meshData = visibilityNode.meshes[i]

        if not meshData.ignoreFillType and (meshData.supportsWrapping or not isWrapped) then
            setVisibility(meshData.node, true)

            setShaderParameter(meshData.node, "wrappingState", wrappingState, 0, 0, 0, false)

            if isWrapped and getHasShaderParameter(meshData.node, "colorScale") then
                setShaderParameter(meshData.node, "colorScale", r, g, b, a, false)
            end
        else
            setVisibility(meshData.node, false)
        end
    end
end
