--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 25.08.2019
Revision: FS22-03

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

ObjectStorage = {}

ObjectStorage.MOD_NAME = g_currentModName
ObjectStorage.MOD_DIR = g_currentModDirectory

ObjectStorage.INPUT_DISABLED_MARKER = g_currentModDirectory .. "shared/marker/markerIcons.i3d"

ObjectStorage.STORAGE_SEND_NUM_BITS = 4
ObjectStorage.OBJECTS_SEND_NUM_BITS = 10
ObjectStorage.TYPE_SEND_NUM_BITS = 9

ObjectStorage.MAX_TRIGGER_WARNING_DISTANCE = 20

local ObjectStorage_mt = Class(ObjectStorage, Object)
local EMPTY_TABLE = {}

InitObjectClass(ObjectStorage, "ObjectStorage")

function ObjectStorage.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".playerTrigger#node", "Trigger node")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".inputTrigger#node", "Trigger node")

    schema:register(XMLValueType.BOOL, basePath .. ".inputTrigger#canDisable", "Allow trigger state change in GUI. If trigger is KINEMATIC it will be reset so any objects in the trigger are added to storage when enabling again", true)
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".inputTrigger#markerLinkNode", "Link node to attach default 'red' tipping disabled marker to")
    ObjectChangeUtil.registerObjectChangeXMLPaths(schema, basePath .. ".inputTrigger") -- Changes are based on input state using standard 'objectChange' options

    schema:register(XMLValueType.STRING, basePath .. ".sharedVisibilityNodes#linkNodesFilename", "Path to i3d file")
    schema:register(XMLValueType.BOOL, basePath .. ".sharedVisibilityNodes#debugPrintsEnabled", "Enable debug logging", false)

    schema:register(XMLValueType.STRING, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#filename", "Path to XML file")
    schema:register(XMLValueType.STRING, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#name", "Name used to identify XML key", "unknown")
    schema:register(XMLValueType.INT, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#variation", "Variation index, allows different layouts to be created for each area if required", 1)
    schema:register(XMLValueType.STRING, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#linkNodes", "Link nodes index", "0")
    schema:register(XMLValueType.STRING, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#fillTypes", "Fill types using this visibility node. If none given then all accepted fill types will be used. If 'UNKNOWN' is used then any accepted fill types without a matching visibility node will use this")

    schema:register(XMLValueType.STRING, basePath .. ".storageAreas#fillTypes", "Fill types to allow from selection screen for area")

    schema:register(XMLValueType.INT, basePath .. ".storageAreas.storageArea(?)#capacity", "When no visibility nodes are given this is the maximum stored objects. Recommended not to exceed 500 > 600 and even less if using multiple storage areas", 50)
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?)#defaultFillType", "Fill type selected on new save or first placement")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".storageAreas.storageArea(?)#cameraNode", "Optional camera that can be viewed from the GUI. If node provided does not have the camera class id then it will be used as a link node instead")

    schema:register(XMLValueType.NODE_INDEX, basePath .. ".storageAreas.storageArea(?).visibilityNodes#node", "Link node for loaded visibility nodes")
    schema:register(XMLValueType.BOOL, basePath .. ".storageAreas.storageArea(?).visibilityNodes#visibleWhenPlacing", "Visibility nodes will be visible when placing", false)
    schema:register(XMLValueType.INT, basePath .. ".storageAreas.storageArea(?).visibilityNodes#variation", "Shared visibility nodes variation to use", 1)

    schema:register(XMLValueType.NODE_INDEX, basePath .. ".storageAreas.storageArea(?).displays.display(?)#node", "Display start node")
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).displays.display(?)#font", "Display font name")
    schema:register(XMLValueType.BOOL, basePath .. ".storageAreas.storageArea(?).displays.display(?)#upperCase", "Display text upper case only", false)
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).displays.display(?)#alignment", "Display text alignment")
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).displays.display(?)#size", "Display text size", 0.03)
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).displays.display(?)#scaleX", "Display text x scale", 1)
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).displays.display(?)#scaleY", "Display text y scale", 1)
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).displays.display(?)#mask", "Display text mask", "0000000")
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).displays.display(?)#emissiveScale", "Display emissive scale", 0.2)
    schema:register(XMLValueType.COLOR, basePath .. ".storageAreas.storageArea(?).displays.display(?)#color", "Display text colour", "0.9 0.9 0.9 1")
    schema:register(XMLValueType.COLOR, basePath .. ".storageAreas.storageArea(?).displays.display(?)#hiddenColor", "Display text hidden colour")
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).displays.display(?)#fillLevelColorType", "Percent & FillLevel options. NONE: No colour change | STANDARD: Text colour will change from 'green > red' | INVERTED: Text colour will change from 'red > green'", "NONE")
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).displays.display(?)#type", "Display type. Options: fillLevel | percent | capacity | title (ENGLISH ONLY)")
    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).displays.display(?)#extraText", "Extra text to be displayed after 'TITLE' only")

    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).configurations.configuration(?)#activeConditionFlag", "Flag to use, if 'unknown' then configuration will be active when no other flag is valid [squarebale | squarebale_xxx | roundbale | roundbale_xxx | Fill type name | unknown]")
    ObjectChangeUtil.registerObjectChangeXMLPaths(schema, basePath .. ".storageAreas.storageArea(?).configurations.configuration(?)")

    schema:register(XMLValueType.STRING, basePath .. ".storageAreas.storageArea(?).fillTypeMaterials.material(?)#fillType", "Fill type name, If 'unknown' is used then this will be used if no valid fill type is found")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".storageAreas.storageArea(?).fillTypeMaterials.material(?)#node", "Node which receives material")
    schema:register(XMLValueType.NODE_INDEX, basePath .. ".storageAreas.storageArea(?).fillTypeMaterials.material(?)#refNode", "Node which provides material")

    -- Sample will play when more than one object is in storage, looping is defined in the XML but is designed to play on loop
    SoundManager.registerSampleXMLPaths(schema, basePath .. ".storageAreas.storageArea(?)", "fillLevelSample")

    -- Animation nodes will be active when more than one object is in storage
    AnimationManager.registerAnimationNodesXMLPaths(schema, basePath .. ".storageAreas.storageArea(?).animationNodes")

    -- Object Changes will apply active state when more than one object is in storage or when capacity is reached depending on the 'activeOnLast' boolean value given
    schema:register(XMLValueType.BOOL, basePath .. ".storageAreas.storageArea(?).fillLevelObjectChanges.objectChange(?)#activeOnLast", "When 'true' active state is applied when the storage reaches capacity otherwise active state is applied when fill level > 0", false)
    ObjectChangeUtil.registerObjectChangeXMLPaths(schema, basePath .. ".storageAreas.storageArea(?).fillLevelObjectChanges")

    ObjectStorageSpawner.registerXMLPaths(schema, basePath .. ".spawnArea")

    -- Sample will play when more than one object is in storage for any storage area, looping is defined in the XML but is designed to play on loop
    SoundManager.registerSampleXMLPaths(schema, basePath, "sharedFillLevelSample")

    -- Animation nodes will be active when more than one object is in storage for any storage area
    AnimationManager.registerAnimationNodesXMLPaths(schema, basePath .. ".sharedAnimationNodes")

    BaleObjectStorage.registerXMLPaths(schema, basePath)
    PalletObjectStorage.registerXMLPaths(schema, basePath)
end

function ObjectStorage.registerSavegameXMLPaths(schema, basePath)
    schema:register(XMLValueType.BOOL, basePath .. "#inputTriggerState", "Current state of the input trigger")

    schema:register(XMLValueType.STRING, basePath .. ".area(?)#defaultFilename", "Default object XML file")
    schema:register(XMLValueType.STRING, basePath .. ".area(?)#commonFilename", "Most common object XML file")
    schema:register(XMLValueType.STRING, basePath .. ".area(?)#fillType", "Current object fill type")
    schema:register(XMLValueType.INT, basePath .. ".area(?)#numObjects", "Current number of stored objects")

    BaleObjectStorage.registerSavegameXMLPaths(schema, basePath)
    PalletObjectStorage.registerSavegameXMLPaths(schema, basePath)
end

function ObjectStorage.new(isServer, isClient, baseDirectory, customEnvironment, customMt)
    local self = Object.new(isServer, isClient, customMt or ObjectStorage_mt)

    self.baseDirectory = baseDirectory
    self.customEnvironment = customEnvironment

    self.owningPlaceable = nil
    self.isOwned = false

    self.name = "Object Storage"

    self.isDeleting = false
    self.isDeleted = false

    self.baleStorage = false
    self.palletStorage = false

    self.availableObjectTypes = {}
    self.acceptedFillTypes = {}
    self.sortedAcceptedTypes = {}

    self.storageAreasByFillType = {}
    self.indexedStorageAreas = {}
    self.storageAreasUpdateTime = {}

    self.lastWarningTime = -1
    self.lastWarningTypeId = -1

    self.inputTriggerState = true
    self.canDisableInputTrigger = true

    self.infoStorageEmpty = {
        title = "-",
        text = g_i18n:getText("infohud_storageIsEmpty")
    }

    self.infoInputTriggerState = {
        accentuate = true,
        title = "",
        text = g_i18n:getText("message_objectStorage_inputTriggerDisabled", ObjectStorage.MOD_NAME)
    }

    return self
end

function ObjectStorage:load(xmlFile, key, components, i3dMappings)
    self.node = components[1].node
    self.i3dMappings = i3dMappings

    if self.owningPlaceable == nil then
        Logging.error("ObjectStorage.owningPlaceable was not set before load()!")

        return false
    end

    self.interactionTrigger = xmlFile:getValue(key .. ".playerTrigger#node", nil, components, i3dMappings)

    if self.interactionTrigger == nil then
        Logging.xmlError(xmlFile, "No player trigger defined!")

        return false
    end

    if not CollisionFlag.getHasFlagSet(self.interactionTrigger, CollisionFlag.TRIGGER_PLAYER) then
        Logging.xmlWarning(xmlFile, "Player trigger '%s.playerTrigger' does not have Bit '%d' (CollisionFlag.TRIGGER_PLAYER) set!", key, CollisionFlag.getBit(CollisionFlag.TRIGGER_PLAYER))
    end

    self.activatable = ObjectStorageActivatable.new(self)

    self.inputTrigger = xmlFile:getValue(key .. ".inputTrigger#node", nil, components, i3dMappings)

    if self.inputTrigger == nil then
        Logging.xmlError(xmlFile, "No input trigger defined!")

        return false
    end

    self.triggerStateChanging = true
    self.canDisableInputTrigger = xmlFile:getValue(key .. ".inputTrigger#canDisable", true)

    if self.canDisableInputTrigger then
        local markerLinkNode = xmlFile:getValue(key .. ".inputTrigger#markerLinkNode", nil, components, i3dMappings)

        if markerLinkNode ~= nil and fileExists(ObjectStorage.INPUT_DISABLED_MARKER) then
            local i3dNode, sharedLoadRequestId = g_i3DManager:loadSharedI3DFile(ObjectStorage.INPUT_DISABLED_MARKER, false, false)

            if self.sharedLoadRequestIds == nil then
                self.sharedLoadRequestIds = {}
            end

            table.insert(self.sharedLoadRequestIds, sharedLoadRequestId)

            if i3dNode ~= 0 then
                local inputDisabledMarker = I3DUtil.indexToObject(i3dNode, "0|0")

                if inputDisabledMarker ~= nil then
                    self.inputDisabledMarker = inputDisabledMarker

                    setVisibility(inputDisabledMarker, false)
                    link(markerLinkNode, inputDisabledMarker)
                end

                delete(i3dNode)
            end
        end

        if xmlFile:hasProperty(key .. ".inputTrigger.objectChange(0)") then
            self.inputTriggerObjectChanges = {}

            ObjectChangeUtil.loadObjectChangeFromXML(xmlFile, key .. ".inputTrigger", self.inputTriggerObjectChanges, components, self)
            ObjectChangeUtil.setObjectChanges(self.inputTriggerObjectChanges, true)
        end
    end

    if not CollisionFlag.getHasFlagSet(self.inputTrigger, CollisionFlag.TRIGGER_DYNAMIC_OBJECT) then
        Logging.xmlError(xmlFile, "Input trigger '%s.inputTrigger' does not have Bit '%d' (CollisionFlag.TRIGGER_DYNAMIC_OBJECT) set!", key, CollisionFlag.getBit(CollisionFlag.TRIGGER_DYNAMIC_OBJECT))
    end

    local spawnAreaKey = key .. ".spawnArea"

    if xmlFile:hasProperty(spawnAreaKey) then
        self.spawnArea = ObjectStorageSpawner.new(self.isServer, self.isClient)

        if not self.spawnArea:load(xmlFile, spawnAreaKey, self, components, i3dMappings) then
            Logging.xmlError(xmlFile, "Unable to load object spawner %s", spawnAreaKey)

            return false
        end
    end

    if self.isClient then
        -- Shared Sample
        if xmlFile:hasProperty(key .. ".sharedFillLevelSample") then
            if self.sharedOperationObjects == nil then
                self.sharedOperationObjects = {}
                self.sharedOperationObjectsActive = false
            end

            self.sharedOperationObjects.fillLevelSample = g_soundManager:loadSampleFromXML(xmlFile, key, "sharedFillLevelSample", self.baseDirectory, components, 0, AudioGroup.ENVIRONMENT, i3dMappings, nil)
        end

        -- Shared Animations
        if xmlFile:hasProperty(key .. ".sharedAnimationNodes") then
            if self.sharedOperationObjects == nil then
                self.sharedOperationObjects = {}
                self.sharedOperationObjectsActive = false
            end

            self.sharedOperationObjects.animationNodes = g_animationManager:loadAnimations(xmlFile, key .. ".sharedAnimationNodes", components, self, i3dMappings)
        end
    end

    return true
end

function ObjectStorage:loadStorageAreas(xmlFile, key, components, i3dMappings, acceptedFillTypes, sortedAcceptedTypes, getAreaAcceptedTypesFunc)
    local hasMultipleAreas = xmlFile:hasProperty(key .. ".storageAreas.storageArea(1)")

    -- If using 10 areas then it is best to limit the storage per area to something like 250.
    -- This is still 2500 objects that need to be synced and does anyone really need that many objects?
    local maxAreas = math.min(2 ^ ObjectStorage.STORAGE_SEND_NUM_BITS - 1, 10)

    -- Limited for multiplayer performance and save game size, 650 is a large limit per area.
    -- This is a possible 6500 objects with 10 areas, that would be silly so do not do this!
    local maxObjects = math.min(2 ^ ObjectStorage.OBJECTS_SEND_NUM_BITS - 1, 650)

    xmlFile:iterate(key .. ".storageAreas.storageArea", function (_, storageAreaKey)
        local areaIndex = #self.indexedStorageAreas + 1

        if areaIndex > maxAreas then
            Logging.xmlWarning(xmlFile, "Maximum storage areas reached, only '%d' areas are supported per building!", maxAreas)

            return false
        end

        local area = {
            objects = {},
            numObjects = 0,
            index = areaIndex
        }

        if hasMultipleAreas and getAreaAcceptedTypesFunc ~= nil then
            area.sortedAcceptedTypes = getAreaAcceptedTypesFunc(storageAreaKey, areaIndex)
        end

        local defaultIndex, defaultFillType = nil, nil
        local acceptedTypes = area.sortedAcceptedTypes or sortedAcceptedTypes

        local fillTypeName = xmlFile:getValue(storageAreaKey .. "#defaultFillType")

        if fillTypeName ~= nil then
            local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

            if fillTypeIndex ~= nil and acceptedFillTypes[fillTypeIndex] then
                for _, acceptedType in ipairs (acceptedTypes) do
                    if acceptedType.fillTypeIndex == fillTypeIndex then
                        defaultIndex = acceptedType.objectTypeIndex
                        defaultFillType = fillTypeIndex

                        break
                    end
                end
            end
        end

        if defaultIndex == nil or defaultFillType == nil then
            defaultIndex = acceptedTypes[1].objectTypeIndex
            defaultFillType = acceptedTypes[1].fillTypeIndex
        end

        local cameraNode = xmlFile:getValue(storageAreaKey .. "#cameraNode", nil, components, i3dMappings)

        if cameraNode ~= nil then
            -- If Class Id is a camera then use this otherwise create one and link
            if getHasClassId(cameraNode, ClassIds.CAMERA) then
                area.cameraNode = cameraNode
            else
                local camera = nil

                for _, storageArea in ipairs (self.indexedStorageAreas) do
                    -- If shared link node then use existing camera
                    if storageArea.cameraNode ~= nil and getParent(storageArea.cameraNode) == cameraNode then
                        camera = storageArea.cameraNode

                        break
                    end
                end

                if camera == nil then
                    camera = createCamera("cameraStorageArea", math.rad(60), 0.15, 6000)
                    link(cameraNode, camera)
                end

                area.cameraNode = camera
            end
        end

        area.visibilityNodesLinkNode = xmlFile:getValue(storageAreaKey .. ".visibilityNodes#node", nil, components, i3dMappings)
        area.visibilityNodesVariation = math.max(xmlFile:getValue(storageAreaKey .. ".visibilityNodes#variation", 1), 1)

        area.defaultMaxObjects = math.min(math.max(xmlFile:getValue(storageAreaKey .. "#capacity", 50), 0), maxObjects)
        area.maxObjects = area.defaultMaxObjects

        area.objectType = self.availableObjectTypes[defaultIndex]
        area.fillTypeIndex = defaultFillType or FillType.UNKNOWN

        if self.isClient then
            area.staticDisplays, area.dynamicDisplays = self:loadDisplays(xmlFile, storageAreaKey, components, i3dMappings)

            area.configurations = self:loadConfigurations(xmlFile, storageAreaKey, components, i3dMappings)

            area.fillTypeMaterials = self:loadFillTypeMaterials(xmlFile, storageAreaKey, components, i3dMappings, acceptedFillTypes)

            area.fillLevelSample = g_soundManager:loadSampleFromXML(xmlFile, storageAreaKey, "fillLevelSample", self.baseDirectory, components, 0, AudioGroup.ENVIRONMENT, i3dMappings, nil)

            if xmlFile:hasProperty(storageAreaKey .. ".animationNodes") then
                area.animationNodes = g_animationManager:loadAnimations(xmlFile, storageAreaKey .. ".animationNodes", components, self, i3dMappings)
            end

            if xmlFile:hasProperty(storageAreaKey .. ".fillLevelObjectChanges") then
                xmlFile:iterate(storageAreaKey .. ".fillLevelObjectChanges.objectChange", function (_, objectChangeKey)
                    local node = xmlFile:getValue(objectChangeKey .. "#node", nil, components, i3dMappings)

                    if node ~= nil then
                        local object = {
                            activeOnLast = xmlFile:getValue(objectChangeKey .. "#activeOnLast", false),
                            node = node
                        }

                        ObjectChangeUtil.loadValuesFromXML(xmlFile, objectChangeKey, node, object, self, components, i3dMappings)

                        if object.activeOnLast then
                            if area.lastObjectChanges == nil then
                                area.lastObjectChanges = {}
                            end

                            table.insert(area.lastObjectChanges, object)
                        else
                            if area.firstObjectChanges == nil then
                                area.firstObjectChanges = {}
                            end

                            table.insert(area.firstObjectChanges, object)
                        end

                        ObjectChangeUtil.setObjectChange(object, false)
                    end
                end)
            end
        end

        self.indexedStorageAreas[areaIndex] = area

        if self.isClient and g_currentMission.gameStarted then
            if area.objectType ~= nil and area.visibilityNodesLinkNode ~= nil then
                area.showVisibilityNodesWhenPlacing = xmlFile:getValue(storageAreaKey .. ".visibilityNodes#visibleWhenPlacing", false)
            end
        end
    end)
end

function ObjectStorage:loadDisplays(xmlFile, key, components, i3dMappings)
    local staticDisplays = nil
    local dynamicDisplays = nil

    local function getFillLevelColourId(fillLevelColourType)
        if fillLevelColourType == "STANDARD" then
            return 1
        elseif fillLevelColourType == "INVERTED" then
            return 2
        end

        return 0
    end

    xmlFile:iterate(key .. ".displays.display", function (_, displayKey)
        local displayNode = xmlFile:getValue(displayKey .. "#node", nil, components, i3dMappings)

        if displayNode ~= nil then
            local fontName = xmlFile:getValue(displayKey .. "#font", "digit"):upper()
            local fontMaterial = g_materialManager:getFontMaterial(fontName, self.customEnvironment)

            if fontMaterial == nil then
                fontMaterial = g_materialManager:getFontMaterial(fontName, ObjectStorage.MOD_NAME)
            end

            if fontMaterial ~= nil then
                local displayType = xmlFile:getValue(displayKey .. "#type", "fillLevel"):upper()
                local titleDisplay = displayType == "TITLE" -- ENGLISH ONLY: The base fonts do not support special characters used by other languages

                if not titleDisplay or g_languageShort == "en" then
                    local display = {
                        displayNode = displayNode,
                        fontMaterial = fontMaterial,
                        upperCase = xmlFile:getValue(displayKey .. "#upperCase", false)
                    }

                    local alignmentStr = xmlFile:getValue(displayKey .. "#alignment", "RIGHT")
                    local alignment = RenderText["ALIGN_" .. alignmentStr:upper()] or RenderText.ALIGN_RIGHT

                    local size = xmlFile:getValue(displayKey .. "#size", 0.03)
                    local scaleX = xmlFile:getValue(displayKey .. "#scaleX", 1)
                    local scaleY = xmlFile:getValue(displayKey .. "#scaleY", 1)
                    local mask = xmlFile:getValue(displayKey .. "#mask", "0000000")
                    local emissiveScale = xmlFile:getValue(displayKey .. "#emissiveScale", 0.2)
                    local colour = xmlFile:getValue(displayKey .. "#color", {0.9, 0.9, 0.9, 1}, true)
                    local hiddenColor = xmlFile:getValue(displayKey .. "#hiddenColor", nil, true)
                    local fillLevelColourType = getFillLevelColourId(xmlFile:getValue(displayKey .. "#fillLevelColorType", "none"):upper())

                    if titleDisplay then
                        local extraText = xmlFile:getValue(displayKey .. "#extraText")

                        if extraText ~= nil then
                            display.extraText = " " .. g_i18n:convertText(extraText, self.customEnvironment)
                        end
                    end

                    display.numChars = mask:len()
                    display.alignment = alignment
                    display.hiddenColor = hiddenColor
                    display.fillLevelColourType = fillLevelColourType
                    display.updateFunction, display.static = self:getDisplayFunction(displayType)

                    display.formatStr, display.formatPrecision = string.maskToFormat(mask)
                    display.characterLine = fontMaterial:createCharacterLine(display.displayNode, display.numChars, size, colour, hiddenColor, emissiveScale, scaleX, scaleY, alignment)

                    if size >= 0.1 then
                        local characters = display.characterLine.characters

                        for i = 1, #characters do
                            setClipDistance(characters[i], 150)
                        end
                    end

                    if display.static then
                        if staticDisplays == nil then
                            staticDisplays = {}
                        end

                        table.insert(staticDisplays, display)
                    else
                        if dynamicDisplays == nil then
                            dynamicDisplays = {}
                        end

                        table.insert(dynamicDisplays, display)
                    end
                end
            end
        end
    end)

    return staticDisplays, dynamicDisplays
end

function ObjectStorage:loadConfigurations(xmlFile, key, components, i3dMappings)
    local configurations

    xmlFile:iterate(key .. ".configurations.configuration", function (_, configurationKey)
        local activeConditionFlag = xmlFile:getValue(configurationKey .. "#activeConditionFlag", "SQUAREBALE"):upper()
        local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(activeConditionFlag)

        local size
        local flag = activeConditionFlag
        local start = activeConditionFlag:find("_")

        if start then
            flag = activeConditionFlag:sub(1, start - 1)
            size = tonumber(activeConditionFlag:sub(start + 1))
        end

        local isRoundbale = flag == "ROUNDBALE"
        local isSquarebale = flag == "SQUAREBALE"

        if fillTypeIndex ~= nil or isRoundbale or isSquarebale then
            local configuration = {
                flag = activeConditionFlag,
                fillTypeIndex = fillTypeIndex,
                isRoundbale = isRoundbale,
                isSquarebale = isSquarebale,
                objectChanges = {},
                size = size
            }

            ObjectChangeUtil.loadObjectChangeFromXML(xmlFile, configurationKey, configuration.objectChanges, components, self)
            ObjectChangeUtil.setObjectChanges(configuration.objectChanges, false)

            if configurations == nil then
                configurations = {}
            end

            table.insert(configurations, configuration)
        else
            Logging.xmlWarning(xmlFile, "Invalid activeConditionFlag '%s' given! Use: squarebale, squarebale_xxx, roundbale, roundbale_xxx or Fill type name", activeConditionFlag)
        end
    end)

    return configurations
end

function ObjectStorage:loadSharedVisibilityNodes(xmlFile, key, acceptedFillTypes, baleStorage)
    local sharedVisibilityNodes = nil

    local linkNodesFilename = ObjectStorageManager.getFilename(xmlFile:getValue(key .. ".sharedVisibilityNodes#linkNodesFilename"), self.baseDirectory)

    if linkNodesFilename ~= nil and fileExists(linkNodesFilename) then
        local linkNodesRoot, sharedLoadRequestId = g_i3DManager:loadSharedI3DFile(linkNodesFilename, false, false)
        local debugPrintsEnabled = xmlFile:getValue(key .. ".sharedVisibilityNodes#debugPrintsEnabled", false)

        if linkNodesRoot == 0 then
            return nil
        end

        xmlFile:iterate(key .. ".sharedVisibilityNodes.visibilityNodes", function (_, visibilityNodesKey)
            local name = xmlFile:getValue(visibilityNodesKey .. "#name", "unknown")
            local linkNodes = xmlFile:getValue(visibilityNodesKey .. "#linkNodes", "0")

            local hasLinkNodes = I3DUtil.indexToObject(linkNodesRoot, linkNodes) ~= nil
            local hasValidXML = false

            local xmlFilename = ObjectStorageManager.getFilename(xmlFile:getValue(visibilityNodesKey .. "#filename"), self.baseDirectory)
            local visibilityNodesXmlFile = XMLFile.loadIfExists("sharedVisibilityNodes", xmlFilename, ObjectStorageVisibilityNodes.xmlSchema)

            if visibilityNodesXmlFile ~= nil then
                visibilityNodesXmlFile:iterate("visibilityNodes.visibilityNode", function (_, sharedKey)
                    if visibilityNodesXmlFile:getValue(sharedKey .. "#name") == name then
                        hasValidXML = true

                        return false
                    end
                end)

                if hasLinkNodes and hasValidXML then
                    local variation = math.max(xmlFile:getValue(visibilityNodesKey .. "#variation", 1), 1)
                    local excludedFillTypes = {}

                    local availableType = {
                        name = name,
                        xmlFilename = xmlFilename,
                        linkNodes = linkNodes,
                        variation = variation,
                        isBackup = false,
                        fillTypes = {}
                    }

                    if baleStorage then
                        local excludedFillTypeNames = xmlFile:getValue(visibilityNodesKey .. "#excludedFillTypes")

                        if excludedFillTypeNames ~= nil then
                            excludedFillTypeNames = string.split(excludedFillTypeNames, " ")

                            for _, fillTypeName in pairs (excludedFillTypeNames) do
                                local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

                                if fillTypeIndex ~= nil then
                                    excludedFillTypes[fillTypeIndex] = true
                                end
                            end
                        end

                        availableType.isRoundbale = xmlFile:getValue(visibilityNodesKey .. "#isRoundbale", false)
                        availableType.width = MathUtil.round(xmlFile:getValue(visibilityNodesKey .. "#width", 0), 2)
                        availableType.height = MathUtil.round(xmlFile:getValue(visibilityNodesKey .. "#height", 0), 2)
                        availableType.length = MathUtil.round(xmlFile:getValue(visibilityNodesKey .. "#length", 0), 2)
                        availableType.diameter = MathUtil.round(xmlFile:getValue(visibilityNodesKey .. "#diameter", 0), 2)
                    end

                    local fillTypeNames = xmlFile:getValue(visibilityNodesKey .. "#fillTypes")
                    local numFillTypes = 0

                    if fillTypeNames ~= nil then
                        if fillTypeNames:upper() ~= "UNKNOWN" then
                            fillTypeNames = string.split(fillTypeNames, " ")

                            for _, fillTypeName in pairs (fillTypeNames) do
                                local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

                                if fillTypeIndex ~= nil and fillTypeIndex ~= FillType.UNKNOWN then
                                    if acceptedFillTypes[fillTypeIndex] then
                                        availableType.fillTypes[fillTypeIndex] = true
                                        numFillTypes = numFillTypes + 1
                                    end
                                end
                            end
                        else
                            availableType.isBackup = true
                            availableType.fillTypes[FillType.UNKNOWN] = true
                            numFillTypes = 1
                        end
                    else
                        for fillTypeIndex, _ in pairs(acceptedFillTypes) do
                            if not excludedFillTypes[fillTypeIndex] then
                                availableType.fillTypes[fillTypeIndex] = true
                                numFillTypes = numFillTypes + 1
                            end
                        end
                    end

                    if numFillTypes > 0 then
                        if sharedVisibilityNodes == nil then
                            sharedVisibilityNodes = {
                                debugPrintsEnabled = debugPrintsEnabled,
                                linkNodesFilename = linkNodesFilename,
                                availableTypes = {}
                            }
                        end

                        table.insert(sharedVisibilityNodes.availableTypes, availableType)
                    end
                else
                    if not hasLinkNodes then
                        Logging.xmlWarning(xmlFile, "File '%s' does not contain a link node group with the name '%s'!", xmlFilename, name)
                    end

                    if not hasValidXML then
                        Logging.xmlWarning(xmlFile, "File '%s' does not contain a '#name' key with the name '%s'!", linkNodesFilename, name)
                    end
                end

                visibilityNodesXmlFile:delete()
            else
                Logging.xmlWarning(xmlFile, "Invalid or no '#filename' given at '%s'!", visibilityNodesKey)
            end
        end)

        delete(linkNodesRoot)

        if sharedLoadRequestId ~= nil then
            if self.sharedLoadRequestIds == nil then
                self.sharedLoadRequestIds = {}
            end

            table.insert(self.sharedLoadRequestIds, sharedLoadRequestId)
        end
    end

    self.sharedVisibilityNodes = sharedVisibilityNodes
end

function ObjectStorage:loadFillTypeMaterials(xmlFile, key, components, i3dMappings, acceptedFillTypes)
    local fillTypeMaterials

    xmlFile:iterate(key .. ".fillTypeMaterials.material", function (_, materialKey)
        local fillTypeName = xmlFile:getValue(materialKey .. "#fillType")

        if fillTypeName ~= nil then
            local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

            if fillTypeIndex ~= nil and (acceptedFillTypes[fillTypeIndex] or fillTypeIndex == FillType.UNKNOWN) then
                local node = xmlFile:getValue(materialKey .. "#node", nil, components, i3dMappings)
                local refNode = xmlFile:getValue(materialKey .. "#refNode", nil, components, i3dMappings)

                if node ~= nil then
                    if fillTypeMaterials == nil then
                        fillTypeMaterials = {}
                    end

                    if fillTypeMaterials[fillTypeIndex] == nil then
                        fillTypeMaterials[fillTypeIndex] = {}
                    end

                    table.insert(fillTypeMaterials[fillTypeIndex], {
                        node = node,
                        refNode = refNode
                    })
                else
                    Logging.xmlWarning(xmlFile, "Missing node or ref node in '%s'", materialKey)
                end
            end
        else
            Logging.xmlWarning(xmlFile, "Missing fill type in '%s'", materialKey)
        end
    end)

    return fillTypeMaterials
end

function ObjectStorage:finalizePlacement()
    self.updateStorageAreas = false
    self.storageAreasUpdateTime = {}

    if self.isServer then
        for _, storageArea in ipairs (self.indexedStorageAreas) do
            if not storageArea.loadedFromSavegame then
                self:setVisibilityNodes(storageArea, self.onStorageObjectTypeChanged, self)
            end

            storageArea.loadedFromSavegame = nil
        end

        if self.inputTrigger ~= nil and self.inputTrigger ~= 0 then
            addTrigger(self.inputTrigger, "inputTriggerCallback", self)
            self.triggerStateChanging = false
        end
    end

    if self.interactionTrigger ~= nil and self.interactionTrigger ~= 0 then
        addTrigger(self.interactionTrigger, "interactionTriggerCallback", self)
    end

    if self.spawnArea ~= nil then
        self.spawnArea:finalizePlacement()
    end
end

function ObjectStorage:delete()
    g_messageCenter:unsubscribeAll(self)

    self.isDeleting = true

    g_objectStorageManager:removeObjectStorage(self)

    g_currentMission.activatableObjectsSystem:removeActivatable(self.activatable)
    self.activatable = nil

    if self.interactionTrigger ~= nil and self.interactionTrigger ~= 0 then
        removeTrigger(self.interactionTrigger)

        self.interactionTrigger = 0
    end

    if self.inputTrigger ~= nil and self.inputTrigger ~= 0 then
        removeTrigger(self.inputTrigger)

        self.inputTrigger = 0
    end

    for _, storageArea in ipairs (self.indexedStorageAreas) do
        if storageArea.visibilityNodes ~= nil then
            storageArea.maxObjects = 0

            if storageArea.visibilityNodes.delete ~= nil then
                storageArea.visibilityNodes:delete()
            end

            storageArea.visibilityNodes = nil
        end

        if storageArea.fillLevelSample ~= nil then
            g_soundManager:deleteSample(storageArea.fillLevelSample)
            storageArea.fillLevelSample = nil
        end

        if storageArea.animationNodes ~= nil then
            g_animationManager:deleteAnimations(storageArea.animationNodes)
            storageArea.animationNodes = nil
        end
    end

    if self.sharedOperationObjects ~= nil then
        if self.sharedOperationObjects.fillLevelSample ~= nil then
            g_soundManager:deleteSample(self.sharedOperationObjects.fillLevelSample)
        end

        if self.sharedOperationObjects.animationNodes ~= nil then
            g_animationManager:deleteAnimations(self.sharedOperationObjects.animationNodes)
        end

        self.sharedOperationObjects = nil
    end

    if self.sharedLoadRequestIds ~= nil then
        for i = 1, #self.sharedLoadRequestIds do
            g_i3DManager:releaseSharedI3DFile(self.sharedLoadRequestIds[i])
        end

        self.sharedLoadRequestIds = nil
    end

    if self.spawnArea ~= nil then
        self.spawnArea:delete()
    end

    self.isDeleting = false
    self.isDeleted = true

    ObjectStorage:superClass().delete(self)
end

function ObjectStorage:readStream(streamId, connection)
    ObjectStorage:superClass().readStream(self, streamId, connection)

    if connection:getIsServer() then
        self:setInputTriggerState(streamReadBool(streamId), true)
    end
end

function ObjectStorage:writeStream(streamId, connection)
    ObjectStorage:superClass().writeStream(self, streamId, connection)

    if not connection:getIsServer() then
        local inputTriggerState = true

        if self.canDisableInputTrigger then
            inputTriggerState = self.inputTriggerState
        end

        streamWriteBool(streamId, inputTriggerState)
    end
end

function ObjectStorage:loadFromXMLFile(xmlFile, key)
    if self.canDisableInputTrigger then
        self:setInputTriggerState(xmlFile:getValue(key .. "#inputTriggerState", self.inputTriggerState), true)
    end
end

function ObjectStorage:saveToXMLFile(xmlFile, key, usedModNames)
    if self.canDisableInputTrigger then
        xmlFile:setValue(key .. "#inputTriggerState", self.inputTriggerState)
    end
end

function ObjectStorage:update(dt)
    if self.isServer then
        if self.lastWarningTime >= 0 then
            self.lastWarningTime = self.lastWarningTime - 1

            self:raiseActive()
        end
    end

    if self.updateStorageAreas then
        self.updateStorageAreas = false

        for storageArea, updateTime in pairs (self.storageAreasUpdateTime) do
            if updateTime > 0 then
                self.updateStorageAreas = true
                self.storageAreasUpdateTime[storageArea] = updateTime - 1
            else
                self:onStorageLevelUpdateFinished(storageArea)
                self.storageAreasUpdateTime[storageArea] = nil
            end
        end

        self:raiseActive()
    end
end

function ObjectStorage:setStorageObjectType(storageIndex, objectTypeIndex, fillTypeIndex, noEventSend)
    local storageArea = self.indexedStorageAreas[storageIndex]
    local objectType = self:getObjectType(objectTypeIndex, fillTypeIndex)

    if storageArea ~= nil and objectType ~= nil and fillTypeIndex ~= nil then
        if storageArea.numObjects <= 0 and objectType.fillTypeIndexs[fillTypeIndex] then
            ObjectStorageSetObjectTypeEvent.sendEvent(self, storageIndex, objectType.index, fillTypeIndex, noEventSend)

            storageArea.objectType = objectType
            storageArea.fillTypeIndex = fillTypeIndex
            storageArea.maxObjects = storageArea.defaultMaxObjects

            self:setVisibilityNodes(storageArea, self.onStorageObjectTypeChanged, self)

            return true
        end
    end

    return false
end

function ObjectStorage:setInputTriggerState(inputTriggerState, noEventSend)
    if self.canDisableInputTrigger then
        self.inputTriggerState = inputTriggerState

        ObjectStorageSetInputTriggerEvent.sendEvent(self, inputTriggerState, noEventSend)

        ObjectChangeUtil.setObjectChanges(self.inputTriggerObjectChanges, inputTriggerState)

        if self.inputDisabledMarker ~= nil then
            setVisibility(self.inputDisabledMarker, not inputTriggerState)
        end

        self:onInputTriggerStateChanged(inputTriggerState)
    else
        self.inputTriggerState = true
    end
end

function ObjectStorage:onInputTriggerStateChanged(active)
    if self.isServer and active and not self.triggerStateChanging and Timer ~= nil then
        local inputTrigger = self.inputTrigger or 0

        if inputTrigger ~= 0 and getRigidBodyType(inputTrigger) == RigidBodyType.KINEMATIC then
            local sx, sy, sz = getScale(inputTrigger)

            if sx == 1 and sy == 1 and sz == 1 then
                self.triggerStateChanging = true
                setScale(inputTrigger, 0, 0, 0)

                Timer.createOneshot(500, function()
                    setScale(inputTrigger, 1, 1, 1)
                    self.triggerStateChanging = false
                end)
            end
        end
    end
end

function ObjectStorage:onStorageObjectTypeChanged(storageArea)
    if self.isClient then
        local fillTypeIndex = storageArea.fillTypeIndex
        local maxObjects = storageArea.maxObjects

        self:updateDisplays(storageArea.staticDisplays, 0, maxObjects, fillTypeIndex, storageArea)
        self:updateDisplays(storageArea.dynamicDisplays, 0, maxObjects, fillTypeIndex, storageArea)

        if storageArea.configurations ~= nil then
            local setUnkownConfigs = true

            for i = 1, #storageArea.configurations do
                local configuration = storageArea.configurations[i]
                local active = false

                if configuration.flag ~= "UNKNOWN" then
                    active = configuration.fillTypeIndex == fillTypeIndex

                    if not active and self.baleStorage and storageArea.objectType ~= nil then
                        local objectType = storageArea.objectType

                        if configuration.isRoundbale and objectType.isRoundbale then
                            active = configuration.size == nil or configuration.size == objectType.diameter * 100
                        elseif configuration.isSquarebale and not objectType.isRoundbale then
                            active = configuration.size == nil or configuration.size == objectType.length * 100
                        end
                    end

                    setUnkownConfigs = false
                end

                ObjectChangeUtil.setObjectChanges(configuration.objectChanges, active)
            end

            if setUnkownConfigs then
                for i = 1, #storageArea.configurations do
                    local configuration = storageArea.configurations[i]

                    if configuration.flag == "UNKNOWN" then
                        ObjectChangeUtil.setObjectChanges(configuration.objectChanges, true)
                    end
                end
            end
        end

        if storageArea.fillTypeMaterials ~= nil then
            local fillTypeMaterials = storageArea.fillTypeMaterials[fillTypeIndex]

            if fillTypeMaterials == nil then
                fillTypeMaterials = storageArea.fillTypeMaterials[FillType.UNKNOWN]
            end

            if fillTypeMaterials ~= nil then
                for _, material in ipairs (fillTypeMaterials) do
                    local materialId = 0

                    if material.refNode ~= nil then
                        materialId = getMaterial(material.refNode, 0) or materialId
                    end

                    if materialId ~= 0 then
                        setMaterial(material.node, materialId, 0)
                        setVisibility(material.node, true)
                    else
                        setVisibility(material.node, false)
                    end
                end
            end
        end
    end

    self:onInputTriggerStateChanged(true)
end

function ObjectStorage:setVisibilityNodes(storageArea, onSetFunction, target)
    if storageArea.visibilityNodes ~= nil then
        -- log("Removing visibility nodes: ", storageArea.visibilityNodes.name)
        -- log("Max objects changed to: ", storageArea.maxObjects)

        if storageArea.visibilityNodes.delete ~= nil then
            storageArea.visibilityNodes:delete()
        end

        storageArea.visibilityNodes = nil
    end

    if self.sharedVisibilityNodes ~= nil and storageArea.visibilityNodesLinkNode ~= nil then
        local visibilityNodes = ObjectStorageVisibilityNodes.new()

        if visibilityNodes:load(storageArea, self) then
            visibilityNodes.valid = true

            storageArea.maxObjects = visibilityNodes:getNumVisibilityNodes()
            storageArea.visibilityNodes = visibilityNodes

            -- log("Adding new visibility nodes: ", visibilityNodes.name)
            -- log("Max objects changed to: ", storageArea.maxObjects)
        else
            storageArea.visibilityNodes = {
                typeId = visibilityNodes.typeId,
                valid = false,
                name = "INVALID"
            }

            visibilityNodes:delete()
        end
    end

    if onSetFunction ~= nil then
        onSetFunction(target, storageArea)
    end
end

function ObjectStorage:updateVisibilityNodes(storageArea, fillLevel, capacity)
    if storageArea.visibilityNodes ~= nil and storageArea.visibilityNodes.onFillLevelChanged ~= nil then
        storageArea.visibilityNodes:onFillLevelChanged(storageArea.objects or EMPTY_TABLE, fillLevel, capacity)
    end
end

function ObjectStorage:updateDisplays(displays, fillLevel, capacity, fillTypeIndex, storageArea)
    if displays ~= nil then
        for _, display in ipairs(displays) do
            if display.updateFunction ~= nil then
                display.updateFunction(display, fillLevel, capacity, fillTypeIndex, storageArea)
            end
        end
    end
end

function ObjectStorage:addToStorage(storageArea, attributes)
    if storageArea == nil or attributes == nil or storageArea.numObjects >= storageArea.maxObjects then
        return false
    end

    local numObjects = #storageArea.objects + 1

    if attributes.xmlFilename == nil or attributes.xmlFilename == "" then
        attributes.xmlFilename = storageArea.objectType and storageArea.objectType.xmlFilename
    end

    storageArea.objects[numObjects] = attributes
    storageArea.numObjects = numObjects

    if self.isServer then
        g_server:broadcastEvent(ObjectStorageAddObjectEvent.new(self, storageArea.index, attributes))
    end

    self:raiseStorageUpdate(storageArea)

    return true
end

function ObjectStorage:removeFromStorage(storageArea)
    local numObjects = storageArea ~= nil and #storageArea.objects or 0

    if numObjects > 0 then
        table.remove(storageArea.objects)

        storageArea.numObjects = #storageArea.objects

        if self.isServer then
            g_server:broadcastEvent(ObjectStorageRemoveObjectEvent.new(self, storageArea.index))
        end

        self:raiseStorageUpdate(storageArea)

        return true
    end

    return false
end

function ObjectStorage:raiseStorageUpdate(storageArea)
    self.storageAreasUpdateTime[storageArea] = 29
    self.updateStorageAreas = true

    self:raiseActive()
end

function ObjectStorage:onStorageLevelUpdateFinished(storageArea)
    if self.isClient then
        local fillLevel = storageArea.numObjects or 0
        local capacity = storageArea.maxObjects or 0
        local isActive = fillLevel > 0

        self:updateVisibilityNodes(storageArea, fillLevel, capacity)
        self:updateDisplays(storageArea.dynamicDisplays, fillLevel, capacity, storageArea.fillTypeIndex, storageArea)

        if storageArea.fillLevelSample ~= nil and storageArea.fillLevelSampleActive ~= isActive then
            storageArea.fillLevelSampleActive = isActive

            if isActive then
                g_soundManager:playSample(storageArea.fillLevelSample)
            else
                g_soundManager:stopSample(storageArea.fillLevelSample)
            end
        end

        if storageArea.animationNodes ~= nil and storageArea.animationNodesActive ~= isActive then
            storageArea.animationNodesActive = isActive

            if isActive then
                g_animationManager:startAnimations(storageArea.animationNodes)
            else
                g_animationManager:stopAnimations(storageArea.animationNodes)
            end
        end

        if self.sharedOperationObjects ~= nil and self.sharedOperationObjectsActive ~= isActive then
            local canChangeState = true

            if not isActive then
                for i = 1, #self.indexedStorageAreas do
                    if (self.indexedStorageAreas[i].numObjects or 0) > 0 then
                        canChangeState = false

                        break
                    end
                end
            end

            if canChangeState then
                self.sharedOperationObjectsActive = isActive

                if self.sharedOperationObjects.fillLevelSample ~= nil then
                    if isActive then
                        g_soundManager:playSample(self.sharedOperationObjects.fillLevelSample)
                    else
                        g_soundManager:stopSample(self.sharedOperationObjects.fillLevelSample)
                    end
                end

                if self.sharedOperationObjects.animationNodes ~= nil then
                    if isActive then
                        g_animationManager:startAnimations(self.sharedOperationObjects.animationNodes)
                    else
                        g_animationManager:stopAnimations(self.sharedOperationObjects.animationNodes)
                    end
                end
            end
        end

        if storageArea.firstObjectChanges ~= nil and storageArea.firstObjectChangesActive ~= isActive then
            storageArea.firstObjectChangesActive = isActive

            ObjectChangeUtil.setObjectChanges(storageArea.firstObjectChanges, isActive)
        end

        if storageArea.lastObjectChanges ~= nil then
            isActive = capacity > 0 and capacity <= fillLevel

            if storageArea.lastObjectChangesActive ~= isActive then
                storageArea.lastObjectChangesActive = isActive

                ObjectChangeUtil.setObjectChanges(storageArea.lastObjectChanges, isActive)
            end
        end

        if g_gui:getIsGuiVisible() then
            g_messageCenter:publish("OBJECT_STORAGE_UPDATE_GUI", self)
        end
    end
end

function ObjectStorage:spawnObjects(storageIndex, numToSpawn, connection)
    local storageArea = self.indexedStorageAreas[storageIndex]

    if storageArea ~= nil and storageArea.numObjects > 0 and self.spawnArea ~= nil then
        if self.isServer then
            self.spawnArea:spawnObjects(numToSpawn, storageArea, nil, nil, nil, connection)
        else
            g_client:getServerConnection():sendEvent(ObjectStorageSpawnObjectsEvent.new(self, storageIndex, numToSpawn))
        end
    end
end

function ObjectStorage:onSpawnObjectsFinished(statusCode, numRequested, numSupplied, connection)
    if self.isServer then
        g_server:broadcastEvent(ObjectStorageSpawnerStatusEvent.new(self, ObjectStorageSpawner.STATUS_UPDATE), true, connection)

        if connection ~= nil and statusCode ~= nil then
            connection:sendEvent(ObjectStorageSpawnerStatusEvent.new(self, statusCode))
        end
    end
end

function ObjectStorage:getMaxSpawnSpace(storageArea, maxObjects, ignoreBlocked)
    if self.spawnArea == nil or storageArea == nil or storageArea.objectType == nil then
        return 0, true, ""
    end

    if not ignoreBlocked then
        local isBlocked, objectName = self.spawnArea:getAreaBlocked()

        if isBlocked then
            return 0, true, objectName
        end
    end

    local objectType = storageArea.objectType
    local _, numFreePlaces = nil, 0

    if self.baleStorage then
        local isRoundbale = objectType.isRoundbale

        local height = isRoundbale and objectType.diameter or objectType.height
        local length = isRoundbale and objectType.diameter or objectType.length

        _, numFreePlaces = self.spawnArea:getFreePlaces(objectType.width, height, length, nil, nil, nil, self.baleStorage, isRoundbale, maxObjects or storageArea.maxObjects)
    else
        _, numFreePlaces = self.spawnArea:getFreePalletPlaces(storageArea.objects, maxObjects or storageArea.maxObjects)
    end

    return numFreePlaces, false, ""
end

function ObjectStorage:raiseWarningMessage(typeId, fillTypeIndex, displayWarnings, size)
    if self.isServer then
        local inputTrigger = self.inputTrigger

        if displayWarnings and inputTrigger ~= nil and inputTrigger ~= 0 then
            if self.lastWarningTime < 0 or self.lastWarningTypeId ~= typeId then
                local farmId = self:getOwnerFarmId()
                local connectionList = {}

                local hasConnections = false
                local invalidFarmMsg = false

                if typeId == ObjectStorageWarningMessageEvent.INVALID_FARM_MESSAGE then
                    invalidFarmMsg = true
                end

                for streamId, connection in pairs(g_server.clientConnections) do
                    local player = g_currentMission.connectionsToPlayer[connection]

                    if player ~= nil and ((player.farmId == farmId and not invalidFarmMsg) or (player.farmId ~= farmId and invalidFarmMsg)) then
                        if player.isControlled then
                            if player.rootNode ~= nil and player.rootNode ~= 0 then
                                if calcDistanceFrom(player.rootNode, inputTrigger) <= ObjectStorage.MAX_TRIGGER_WARNING_DISTANCE then
                                    connectionList[streamId] = connection
                                    hasConnections = true
                                end
                            end
                        elseif g_currentMission.controlledVehicles ~= nil then
                            for _, vehicle in pairs (g_currentMission.controlledVehicles) do
                                if vehicle ~= nil and vehicle.owner == connection then
                                    if vehicle.rootNode ~= nil and vehicle.rootNode ~= 0 then
                                        if calcDistanceFrom(vehicle.rootNode, inputTrigger) <= ObjectStorage.MAX_TRIGGER_WARNING_DISTANCE then
                                            connectionList[streamId] = connection
                                            hasConnections = true
                                        end
                                    end

                                    break
                                end
                            end
                        end
                    end
                end

                if hasConnections then
                    g_server:broadcastEvent(ObjectStorageWarningMessageEvent.new(self, typeId, fillTypeIndex, size), true, nil, nil, nil, connectionList)
                end

                self.lastWarningTime = 40
                self.lastWarningTypeId = typeId
            end
        end

        self:raiseActive()
    end
end

function ObjectStorage:displayWarningMessage(typeId, fillTypeIndex, size)
    if typeId == ObjectStorageWarningMessageEvent.INVALID_FARM_MESSAGE then
        g_currentMission:showBlinkingWarning(g_i18n:getText("warning_youDontHaveAccessToThis"))
    elseif typeId == ObjectStorageWarningMessageEvent.INVALID_SIZE_MESSAGE then
        if self.baleStorage then
            if size ~= nil and size > 0 then
                g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("warning_notAcceptedHere"), string.format("%s (%dcm)", g_i18n:getText("infohud_bale"), size)))
            else
                g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("warning_notAcceptedHere"), string.format("%s (SIZE)", g_i18n:getText("infohud_bale"), size)))
            end
        else
            g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("warning_notAcceptedHere"), string.format("%s (SIZE)", g_i18n:getText("infohud_pallet"))))
        end
    elseif fillTypeIndex ~= FillType.UNKNOWN then
        local title = g_fillTypeManager:getFillTypeTitleByIndex(fillTypeIndex)

        if title ~= nil then
            if typeId == ObjectStorageWarningMessageEvent.INVALID_FILLTYPE_MESSAGE then
                g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("warning_notAcceptedHere"), title))
            elseif typeId == ObjectStorageWarningMessageEvent.NO_SPACE_MESSAGE then
                g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("warning_noMoreFreeCapacity"), title))
            end
        end
    end
end

function ObjectStorage:setOwnerFarmId(farmId, noEventSend)
    if self.owningPlaceable.propertyState ~= Placeable.PROPERTY_STATE_CONSTRUCTION_PREVIEW then
        g_objectStorageManager:removeObjectStorage(self)

        ObjectStorage:superClass().setOwnerFarmId(self, farmId, noEventSend)

        g_objectStorageManager:addObjectStorage(self)
    else
        ObjectStorage:superClass().setOwnerFarmId(self, farmId, noEventSend)

        for _, storageArea in ipairs (self.indexedStorageAreas) do
            if storageArea.showVisibilityNodesWhenPlacing then
                storageArea.showVisibilityNodesWhenPlacing = nil

                self:setVisibilityNodes(storageArea, function ()
                    if storageArea.visibilityNodes ~= nil then
                        storageArea.visibilityNodes:setPlacementActive()
                    end
                end)
            end
        end
    end
end

function ObjectStorage:onObjectStorageAdded()
end

function ObjectStorage:onSetName(name)
    if g_objectStorageManager ~= nil then
        g_objectStorageManager:sortFarmStorages(self:getOwnerFarmId())

        g_messageCenter:publish("OBJECT_STORAGE_UPDATE_GUI", self, true)
    end
end

function ObjectStorage:getName()
    local owner = self.owningPlaceable

    if owner.name == nil and owner.storeItem ~= nil then
        return owner.storeItem.name or self.name
    end

    return owner.name or self.name
end

function ObjectStorage:getHotspot(storageArea)
    if storageArea ~= nil and storageArea.hotspot ~= nil then
        return storageArea.hotspot
    end

    if self.owningPlaceable.getHotspot ~= nil then
        return self.owningPlaceable:getHotspot()
    end

    return nil
end

function ObjectStorage:openMenu()
    g_gui:showGui("InGameMenu")
    g_messageCenter:publishDelayed("OPEN_OBJECT_STORAGE_SCREEN", self)
end

function ObjectStorage:updateInfo(infoTable)
    table.insert(infoTable, self.infoStorageEmpty)
end

function ObjectStorage:inputTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
end

function ObjectStorage:interactionTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if (onEnter or onLeave) and g_currentMission.player and g_currentMission.player.rootNode == otherId then
        if onEnter then
            g_currentMission.activatableObjectsSystem:addActivatable(self.activatable)
        else
            g_currentMission.activatableObjectsSystem:removeActivatable(self.activatable)
        end
    end
end

function ObjectStorage:getValidStorageArea(fillTypeIndex, objectTypeIndex, displayWarnings)
    -- Find valid storage area for given fill type
    local storageAreas = self.storageAreasByFillType[fillTypeIndex]

    if storageAreas == nil then
        self:raiseWarningMessage(ObjectStorageWarningMessageEvent.INVALID_FILLTYPE_MESSAGE, fillTypeIndex, displayWarnings, 0)

        return nil
    end

    -- Bales only: Find valid object type (SIZE) from list of available fill type storage areas
    if self.baleStorage then
        storageAreas = storageAreas[objectTypeIndex]

        if storageAreas == nil then
            local size = 0

            if g_objectStorageManager ~= nil and g_objectStorageManager.getBaleSizeByObjectTypeIndex ~= nil then
                size = g_objectStorageManager:getBaleSizeByObjectTypeIndex(objectTypeIndex)
            end

            self:raiseWarningMessage(ObjectStorageWarningMessageEvent.INVALID_SIZE_MESSAGE, fillTypeIndex, displayWarnings, size)

            return nil
        end
    end

    local firstValidStorageArea = nil

    for i = 1, #storageAreas do
        local storageArea = storageAreas[i]

        -- Continue filling and storage with objects or use the first found storage
        if storageArea.numObjects < storageArea.maxObjects then
            if storageArea.numObjects > 0 then
                return storageArea
            end

            if firstValidStorageArea == nil then
                firstValidStorageArea = storageArea
            end
        end
    end

    if firstValidStorageArea ~= nil then
        return firstValidStorageArea
    end

    -- If there is no valid storage then they all must be full
    self:raiseWarningMessage(ObjectStorageWarningMessageEvent.NO_SPACE_MESSAGE, fillTypeIndex, displayWarnings, 0)
end

function ObjectStorage:getAcceptedType(index, acceptedTypes)
    if acceptedTypes ~= nil then
        return acceptedTypes[index]
    end

    return self.sortedAcceptedTypes[index]
end

function ObjectStorage:getObjectType(index, fillTypeIndex)
    return self.availableObjectTypes[index]
end

function ObjectStorage.getCommonFilename(objects, defaultFilename)
    local numObjects = #objects

    if numObjects == 1 then
        return objects[1].xmlFilename
    end

    if numObjects > 2 then
        local xmlFilename = defaultFilename
        local filenames, lastCount = {}, 0

        for i = 1, numObjects do
            local filename = objects[i].xmlFilename

            if filenames[filename] == nil then
                filenames[filename] = 0
            end

            filenames[filename] = filenames[filename] + 1
        end

        for filename, count in pairs (filenames) do
            if lastCount < count then
                lastCount = count
                xmlFilename = filename
            end
        end

        return xmlFilename -- Use the filename with the most objects to save writing unnecessary info to the XML
    end

    return defaultFilename
end

function ObjectStorage:getDisplayFunction(displayType)
    if displayType == "CAPACITY" then
        return function(display, _, capacity)
            if capacity ~= display.lastValue then
                local int, floatPart = math.modf(capacity)
                local value = string.format(display.formatStr, int, math.abs(math.floor((floatPart + 1e-06) * 10 ^ display.formatPrecision)))

                display.fontMaterial:updateCharacterLine(display.characterLine, value)
                display.lastValue = capacity
            end
        end, true
    elseif displayType == "TITLE" then
        return function(display, _, _, fillTypeIndex)
            if fillTypeIndex ~= nil and fillTypeIndex ~= display.lastValue then
                local fillTypeTitle = g_fillTypeManager:getFillTypeTitleByIndex(fillTypeIndex)

                if fillTypeTitle ~= nil then
                    if display.extraText then
                        fillTypeTitle = fillTypeTitle .. display.extraText
                    end

                    if display.upperCase then
                        fillTypeTitle = fillTypeTitle:upper()
                    end

                    local titleLength = #fillTypeTitle
                    local remaining = display.numChars - titleLength

                    if remaining > 0 then
                        local textStart, textEnd = "", ""

                        if display.alignment == RenderText.ALIGN_CENTER then
                            for i = 1, remaining do
                                if i % 2 == 0 then
                                    textEnd = textEnd .. "|"
                                else
                                    textStart = textStart .. "|"
                                end
                            end
                        else
                            for i = 1, remaining do
                                if display.alignment == RenderText.ALIGN_LEFT then
                                    textEnd = textEnd .. "|"
                                else
                                    textStart = textStart .. "|"
                                end
                            end
                        end

                        fillTypeTitle = textStart .. fillTypeTitle .. textEnd
                    end

                    display.fontMaterial:updateCharacterLine(display.characterLine, fillTypeTitle)
                end

                display.lastValue = fillTypeIndex
            end
        end, true
    elseif displayType == "PERCENT" then
        return function(display, fillLevel, capacity)
            local percent = capacity > 0 and (fillLevel / capacity) or 0

            if percent ~= display.lastValue then
                local int, floatPart = math.modf(percent * 100)
                local value = string.format(display.formatStr, int, math.abs(math.floor((floatPart + 1e-06) * 10 ^ display.formatPrecision)))

                if display.fillLevelColourType == 0 then
                    display.fontMaterial:updateCharacterLine(display.characterLine, value)
                else
                    local textColor = display.characterLine.textColor

                    if display.fillLevelColourType == 2 then
                        textColor[1], textColor[2], textColor[3] = MathUtil.lerp3(0, 1, 0, 1, 0, 0, percent)
                    else
                        textColor[1], textColor[2], textColor[3] = MathUtil.lerp3(1, 0, 0, 0, 1, 0, percent)
                    end

                    if display.hiddenColor == nil then
                        for i = 1, #display.characterLine.characters do
                            display.fontMaterial:setFontCharacterColor(display.characterLine.characters[i], textColor[1], textColor[2], textColor[3])
                        end
                    end

                    display.fontMaterial:updateCharacterLine(display.characterLine, value)
                end

                display.lastValue = percent
            end
        end, false
    end

    return function(display, fillLevel, capacity)
        if fillLevel ~= display.lastValue then
            local int, floatPart = math.modf(fillLevel)
            local value = string.format(display.formatStr, int, math.abs(math.floor((floatPart + 1e-06) * 10 ^ display.formatPrecision)))

            if display.fillLevelColourType == 0 then
                display.fontMaterial:updateCharacterLine(display.characterLine, value)
            else
                local textColor = display.characterLine.textColor
                local alpha = capacity > 0 and (fillLevel / capacity) or 0

                if display.fillLevelColourType == 2 then
                    textColor[1], textColor[2], textColor[3] = MathUtil.lerp3(0, 1, 0, 1, 0, 0, alpha)
                else
                    textColor[1], textColor[2], textColor[3] = MathUtil.lerp3(1, 0, 0, 0, 1, 0, alpha)
                end

                if display.hiddenColor == nil then
                    for i = 1, #display.characterLine.characters do
                        display.fontMaterial:setFontCharacterColor(display.characterLine.characters[i], textColor[1], textColor[2], textColor[3])
                    end
                end

                display.fontMaterial:updateCharacterLine(display.characterLine, value)
            end

            display.lastValue = fillLevel
        end
    end, false
end

ObjectStorageActivatable = {}

local ObjectStorageActivatable_mt = Class(ObjectStorageActivatable)

function ObjectStorageActivatable.new(objectStorage)
    local self = setmetatable({}, ObjectStorageActivatable_mt)

    self.objectStorage = objectStorage
    self.activateText = g_i18n:getText("input_MENU")

    return self
end

function ObjectStorageActivatable:getHasAccess(farmId)
    return farmId == self.objectStorage:getOwnerFarmId() -- No contractor access at this stage. Global UI would get confusing if you could see other farms storage buildings when contracting
    -- return g_currentMission.accessHandler:canFarmAccessOtherId(farmId, self.objectStorage:getOwnerFarmId())
end

function ObjectStorageActivatable:getDistance(x, y, z)
    local interactionTrigger = self.objectStorage.interactionTrigger

    if interactionTrigger ~= nil and interactionTrigger ~= 0 then
        local tx, _, tz = getWorldTranslation(interactionTrigger)

        return MathUtil.getPointPointDistance(tx, tz, x, z)
    end

    return math.huge
end

function ObjectStorageActivatable:run()
    self.objectStorage:openMenu()
end
